"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeleteInsideWord = exports.DeleteWordRight = exports.DeleteWordEndRight = exports.DeleteWordStartRight = exports.DeleteWordLeft = exports.DeleteWordEndLeft = exports.DeleteWordStartLeft = exports.DeleteWordRightCommand = exports.DeleteWordLeftCommand = exports.DeleteWordCommand = exports.CursorWordAccessibilityRightSelect = exports.CursorWordAccessibilityRight = exports.CursorWordRightSelect = exports.CursorWordEndRightSelect = exports.CursorWordStartRightSelect = exports.CursorWordRight = exports.CursorWordEndRight = exports.CursorWordStartRight = exports.CursorWordAccessibilityLeftSelect = exports.CursorWordAccessibilityLeft = exports.CursorWordLeftSelect = exports.CursorWordEndLeftSelect = exports.CursorWordStartLeftSelect = exports.CursorWordLeft = exports.CursorWordEndLeft = exports.CursorWordStartLeft = exports.WordRightCommand = exports.WordLeftCommand = exports.MoveWordCommand = void 0;
const editorExtensions_1 = require("../../../browser/editorExtensions");
const replaceCommand_1 = require("../../../common/commands/replaceCommand");
const editorOptions_1 = require("../../../common/config/editorOptions");
const cursorCommon_1 = require("../../../common/cursorCommon");
const cursorWordOperations_1 = require("../../../common/cursor/cursorWordOperations");
const wordCharacterClassifier_1 = require("../../../common/core/wordCharacterClassifier");
const position_1 = require("../../../common/core/position");
const range_1 = require("../../../common/core/range");
const selection_1 = require("../../../common/core/selection");
const editorContextKeys_1 = require("../../../common/editorContextKeys");
const languageConfigurationRegistry_1 = require("../../../common/languages/languageConfigurationRegistry");
const nls = require("../../../../nls");
const accessibility_1 = require("../../../../platform/accessibility/common/accessibility");
const contextkey_1 = require("../../../../platform/contextkey/common/contextkey");
const contextkeys_1 = require("../../../../platform/contextkey/common/contextkeys");
class MoveWordCommand extends editorExtensions_1.EditorCommand {
    constructor(opts) {
        super(opts);
        this._inSelectionMode = opts.inSelectionMode;
        this._wordNavigationType = opts.wordNavigationType;
    }
    runEditorCommand(accessor, editor, args) {
        if (!editor.hasModel()) {
            return;
        }
        const wordSeparators = (0, wordCharacterClassifier_1.getMapForWordSeparators)(editor.getOption(119 /* EditorOption.wordSeparators */));
        const model = editor.getModel();
        const selections = editor.getSelections();
        const result = selections.map((sel) => {
            const inPosition = new position_1.Position(sel.positionLineNumber, sel.positionColumn);
            const outPosition = this._move(wordSeparators, model, inPosition, this._wordNavigationType);
            return this._moveTo(sel, outPosition, this._inSelectionMode);
        });
        model.pushStackElement();
        editor._getViewModel().setCursorStates('moveWordCommand', 3 /* CursorChangeReason.Explicit */, result.map(r => cursorCommon_1.CursorState.fromModelSelection(r)));
        if (result.length === 1) {
            const pos = new position_1.Position(result[0].positionLineNumber, result[0].positionColumn);
            editor.revealPosition(pos, 0 /* ScrollType.Smooth */);
        }
    }
    _moveTo(from, to, inSelectionMode) {
        if (inSelectionMode) {
            // move just position
            return new selection_1.Selection(from.selectionStartLineNumber, from.selectionStartColumn, to.lineNumber, to.column);
        }
        else {
            // move everything
            return new selection_1.Selection(to.lineNumber, to.column, to.lineNumber, to.column);
        }
    }
}
exports.MoveWordCommand = MoveWordCommand;
class WordLeftCommand extends MoveWordCommand {
    _move(wordSeparators, model, position, wordNavigationType) {
        return cursorWordOperations_1.WordOperations.moveWordLeft(wordSeparators, model, position, wordNavigationType);
    }
}
exports.WordLeftCommand = WordLeftCommand;
class WordRightCommand extends MoveWordCommand {
    _move(wordSeparators, model, position, wordNavigationType) {
        return cursorWordOperations_1.WordOperations.moveWordRight(wordSeparators, model, position, wordNavigationType);
    }
}
exports.WordRightCommand = WordRightCommand;
class CursorWordStartLeft extends WordLeftCommand {
    constructor() {
        super({
            inSelectionMode: false,
            wordNavigationType: 0 /* WordNavigationType.WordStart */,
            id: 'cursorWordStartLeft',
            precondition: undefined
        });
    }
}
exports.CursorWordStartLeft = CursorWordStartLeft;
class CursorWordEndLeft extends WordLeftCommand {
    constructor() {
        super({
            inSelectionMode: false,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'cursorWordEndLeft',
            precondition: undefined
        });
    }
}
exports.CursorWordEndLeft = CursorWordEndLeft;
class CursorWordLeft extends WordLeftCommand {
    constructor() {
        var _a;
        super({
            inSelectionMode: false,
            wordNavigationType: 1 /* WordNavigationType.WordStartFast */,
            id: 'cursorWordLeft',
            precondition: undefined,
            kbOpts: {
                kbExpr: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.textInputFocus, (_a = contextkey_1.ContextKeyExpr.and(accessibility_1.CONTEXT_ACCESSIBILITY_MODE_ENABLED, contextkeys_1.IsWindowsContext)) === null || _a === void 0 ? void 0 : _a.negate()),
                primary: 2048 /* KeyMod.CtrlCmd */ | 15 /* KeyCode.LeftArrow */,
                mac: { primary: 512 /* KeyMod.Alt */ | 15 /* KeyCode.LeftArrow */ },
                weight: 100 /* KeybindingWeight.EditorContrib */
            }
        });
    }
}
exports.CursorWordLeft = CursorWordLeft;
class CursorWordStartLeftSelect extends WordLeftCommand {
    constructor() {
        super({
            inSelectionMode: true,
            wordNavigationType: 0 /* WordNavigationType.WordStart */,
            id: 'cursorWordStartLeftSelect',
            precondition: undefined
        });
    }
}
exports.CursorWordStartLeftSelect = CursorWordStartLeftSelect;
class CursorWordEndLeftSelect extends WordLeftCommand {
    constructor() {
        super({
            inSelectionMode: true,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'cursorWordEndLeftSelect',
            precondition: undefined
        });
    }
}
exports.CursorWordEndLeftSelect = CursorWordEndLeftSelect;
class CursorWordLeftSelect extends WordLeftCommand {
    constructor() {
        var _a;
        super({
            inSelectionMode: true,
            wordNavigationType: 1 /* WordNavigationType.WordStartFast */,
            id: 'cursorWordLeftSelect',
            precondition: undefined,
            kbOpts: {
                kbExpr: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.textInputFocus, (_a = contextkey_1.ContextKeyExpr.and(accessibility_1.CONTEXT_ACCESSIBILITY_MODE_ENABLED, contextkeys_1.IsWindowsContext)) === null || _a === void 0 ? void 0 : _a.negate()),
                primary: 2048 /* KeyMod.CtrlCmd */ | 1024 /* KeyMod.Shift */ | 15 /* KeyCode.LeftArrow */,
                mac: { primary: 512 /* KeyMod.Alt */ | 1024 /* KeyMod.Shift */ | 15 /* KeyCode.LeftArrow */ },
                weight: 100 /* KeybindingWeight.EditorContrib */
            }
        });
    }
}
exports.CursorWordLeftSelect = CursorWordLeftSelect;
// Accessibility navigation commands should only be enabled on windows since they are tuned to what NVDA expects
class CursorWordAccessibilityLeft extends WordLeftCommand {
    constructor() {
        super({
            inSelectionMode: false,
            wordNavigationType: 3 /* WordNavigationType.WordAccessibility */,
            id: 'cursorWordAccessibilityLeft',
            precondition: undefined
        });
    }
    _move(_, model, position, wordNavigationType) {
        return super._move((0, wordCharacterClassifier_1.getMapForWordSeparators)(editorOptions_1.EditorOptions.wordSeparators.defaultValue), model, position, wordNavigationType);
    }
}
exports.CursorWordAccessibilityLeft = CursorWordAccessibilityLeft;
class CursorWordAccessibilityLeftSelect extends WordLeftCommand {
    constructor() {
        super({
            inSelectionMode: true,
            wordNavigationType: 3 /* WordNavigationType.WordAccessibility */,
            id: 'cursorWordAccessibilityLeftSelect',
            precondition: undefined
        });
    }
    _move(_, model, position, wordNavigationType) {
        return super._move((0, wordCharacterClassifier_1.getMapForWordSeparators)(editorOptions_1.EditorOptions.wordSeparators.defaultValue), model, position, wordNavigationType);
    }
}
exports.CursorWordAccessibilityLeftSelect = CursorWordAccessibilityLeftSelect;
class CursorWordStartRight extends WordRightCommand {
    constructor() {
        super({
            inSelectionMode: false,
            wordNavigationType: 0 /* WordNavigationType.WordStart */,
            id: 'cursorWordStartRight',
            precondition: undefined
        });
    }
}
exports.CursorWordStartRight = CursorWordStartRight;
class CursorWordEndRight extends WordRightCommand {
    constructor() {
        var _a;
        super({
            inSelectionMode: false,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'cursorWordEndRight',
            precondition: undefined,
            kbOpts: {
                kbExpr: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.textInputFocus, (_a = contextkey_1.ContextKeyExpr.and(accessibility_1.CONTEXT_ACCESSIBILITY_MODE_ENABLED, contextkeys_1.IsWindowsContext)) === null || _a === void 0 ? void 0 : _a.negate()),
                primary: 2048 /* KeyMod.CtrlCmd */ | 17 /* KeyCode.RightArrow */,
                mac: { primary: 512 /* KeyMod.Alt */ | 17 /* KeyCode.RightArrow */ },
                weight: 100 /* KeybindingWeight.EditorContrib */
            }
        });
    }
}
exports.CursorWordEndRight = CursorWordEndRight;
class CursorWordRight extends WordRightCommand {
    constructor() {
        super({
            inSelectionMode: false,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'cursorWordRight',
            precondition: undefined
        });
    }
}
exports.CursorWordRight = CursorWordRight;
class CursorWordStartRightSelect extends WordRightCommand {
    constructor() {
        super({
            inSelectionMode: true,
            wordNavigationType: 0 /* WordNavigationType.WordStart */,
            id: 'cursorWordStartRightSelect',
            precondition: undefined
        });
    }
}
exports.CursorWordStartRightSelect = CursorWordStartRightSelect;
class CursorWordEndRightSelect extends WordRightCommand {
    constructor() {
        var _a;
        super({
            inSelectionMode: true,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'cursorWordEndRightSelect',
            precondition: undefined,
            kbOpts: {
                kbExpr: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.textInputFocus, (_a = contextkey_1.ContextKeyExpr.and(accessibility_1.CONTEXT_ACCESSIBILITY_MODE_ENABLED, contextkeys_1.IsWindowsContext)) === null || _a === void 0 ? void 0 : _a.negate()),
                primary: 2048 /* KeyMod.CtrlCmd */ | 1024 /* KeyMod.Shift */ | 17 /* KeyCode.RightArrow */,
                mac: { primary: 512 /* KeyMod.Alt */ | 1024 /* KeyMod.Shift */ | 17 /* KeyCode.RightArrow */ },
                weight: 100 /* KeybindingWeight.EditorContrib */
            }
        });
    }
}
exports.CursorWordEndRightSelect = CursorWordEndRightSelect;
class CursorWordRightSelect extends WordRightCommand {
    constructor() {
        super({
            inSelectionMode: true,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'cursorWordRightSelect',
            precondition: undefined
        });
    }
}
exports.CursorWordRightSelect = CursorWordRightSelect;
class CursorWordAccessibilityRight extends WordRightCommand {
    constructor() {
        super({
            inSelectionMode: false,
            wordNavigationType: 3 /* WordNavigationType.WordAccessibility */,
            id: 'cursorWordAccessibilityRight',
            precondition: undefined
        });
    }
    _move(_, model, position, wordNavigationType) {
        return super._move((0, wordCharacterClassifier_1.getMapForWordSeparators)(editorOptions_1.EditorOptions.wordSeparators.defaultValue), model, position, wordNavigationType);
    }
}
exports.CursorWordAccessibilityRight = CursorWordAccessibilityRight;
class CursorWordAccessibilityRightSelect extends WordRightCommand {
    constructor() {
        super({
            inSelectionMode: true,
            wordNavigationType: 3 /* WordNavigationType.WordAccessibility */,
            id: 'cursorWordAccessibilityRightSelect',
            precondition: undefined
        });
    }
    _move(_, model, position, wordNavigationType) {
        return super._move((0, wordCharacterClassifier_1.getMapForWordSeparators)(editorOptions_1.EditorOptions.wordSeparators.defaultValue), model, position, wordNavigationType);
    }
}
exports.CursorWordAccessibilityRightSelect = CursorWordAccessibilityRightSelect;
class DeleteWordCommand extends editorExtensions_1.EditorCommand {
    constructor(opts) {
        super(opts);
        this._whitespaceHeuristics = opts.whitespaceHeuristics;
        this._wordNavigationType = opts.wordNavigationType;
    }
    runEditorCommand(accessor, editor, args) {
        const languageConfigurationService = accessor.get(languageConfigurationRegistry_1.ILanguageConfigurationService);
        if (!editor.hasModel()) {
            return;
        }
        const wordSeparators = (0, wordCharacterClassifier_1.getMapForWordSeparators)(editor.getOption(119 /* EditorOption.wordSeparators */));
        const model = editor.getModel();
        const selections = editor.getSelections();
        const autoClosingBrackets = editor.getOption(5 /* EditorOption.autoClosingBrackets */);
        const autoClosingQuotes = editor.getOption(8 /* EditorOption.autoClosingQuotes */);
        const autoClosingPairs = languageConfigurationService.getLanguageConfiguration(model.getLanguageId()).getAutoClosingPairs();
        const viewModel = editor._getViewModel();
        const commands = selections.map((sel) => {
            const deleteRange = this._delete({
                wordSeparators,
                model,
                selection: sel,
                whitespaceHeuristics: this._whitespaceHeuristics,
                autoClosingDelete: editor.getOption(6 /* EditorOption.autoClosingDelete */),
                autoClosingBrackets,
                autoClosingQuotes,
                autoClosingPairs,
                autoClosedCharacters: viewModel.getCursorAutoClosedCharacters()
            }, this._wordNavigationType);
            return new replaceCommand_1.ReplaceCommand(deleteRange, '');
        });
        editor.pushUndoStop();
        editor.executeCommands(this.id, commands);
        editor.pushUndoStop();
    }
}
exports.DeleteWordCommand = DeleteWordCommand;
class DeleteWordLeftCommand extends DeleteWordCommand {
    _delete(ctx, wordNavigationType) {
        const r = cursorWordOperations_1.WordOperations.deleteWordLeft(ctx, wordNavigationType);
        if (r) {
            return r;
        }
        return new range_1.Range(1, 1, 1, 1);
    }
}
exports.DeleteWordLeftCommand = DeleteWordLeftCommand;
class DeleteWordRightCommand extends DeleteWordCommand {
    _delete(ctx, wordNavigationType) {
        const r = cursorWordOperations_1.WordOperations.deleteWordRight(ctx, wordNavigationType);
        if (r) {
            return r;
        }
        const lineCount = ctx.model.getLineCount();
        const maxColumn = ctx.model.getLineMaxColumn(lineCount);
        return new range_1.Range(lineCount, maxColumn, lineCount, maxColumn);
    }
}
exports.DeleteWordRightCommand = DeleteWordRightCommand;
class DeleteWordStartLeft extends DeleteWordLeftCommand {
    constructor() {
        super({
            whitespaceHeuristics: false,
            wordNavigationType: 0 /* WordNavigationType.WordStart */,
            id: 'deleteWordStartLeft',
            precondition: editorContextKeys_1.EditorContextKeys.writable
        });
    }
}
exports.DeleteWordStartLeft = DeleteWordStartLeft;
class DeleteWordEndLeft extends DeleteWordLeftCommand {
    constructor() {
        super({
            whitespaceHeuristics: false,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'deleteWordEndLeft',
            precondition: editorContextKeys_1.EditorContextKeys.writable
        });
    }
}
exports.DeleteWordEndLeft = DeleteWordEndLeft;
class DeleteWordLeft extends DeleteWordLeftCommand {
    constructor() {
        super({
            whitespaceHeuristics: true,
            wordNavigationType: 0 /* WordNavigationType.WordStart */,
            id: 'deleteWordLeft',
            precondition: editorContextKeys_1.EditorContextKeys.writable,
            kbOpts: {
                kbExpr: editorContextKeys_1.EditorContextKeys.textInputFocus,
                primary: 2048 /* KeyMod.CtrlCmd */ | 1 /* KeyCode.Backspace */,
                mac: { primary: 512 /* KeyMod.Alt */ | 1 /* KeyCode.Backspace */ },
                weight: 100 /* KeybindingWeight.EditorContrib */
            }
        });
    }
}
exports.DeleteWordLeft = DeleteWordLeft;
class DeleteWordStartRight extends DeleteWordRightCommand {
    constructor() {
        super({
            whitespaceHeuristics: false,
            wordNavigationType: 0 /* WordNavigationType.WordStart */,
            id: 'deleteWordStartRight',
            precondition: editorContextKeys_1.EditorContextKeys.writable
        });
    }
}
exports.DeleteWordStartRight = DeleteWordStartRight;
class DeleteWordEndRight extends DeleteWordRightCommand {
    constructor() {
        super({
            whitespaceHeuristics: false,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'deleteWordEndRight',
            precondition: editorContextKeys_1.EditorContextKeys.writable
        });
    }
}
exports.DeleteWordEndRight = DeleteWordEndRight;
class DeleteWordRight extends DeleteWordRightCommand {
    constructor() {
        super({
            whitespaceHeuristics: true,
            wordNavigationType: 2 /* WordNavigationType.WordEnd */,
            id: 'deleteWordRight',
            precondition: editorContextKeys_1.EditorContextKeys.writable,
            kbOpts: {
                kbExpr: editorContextKeys_1.EditorContextKeys.textInputFocus,
                primary: 2048 /* KeyMod.CtrlCmd */ | 20 /* KeyCode.Delete */,
                mac: { primary: 512 /* KeyMod.Alt */ | 20 /* KeyCode.Delete */ },
                weight: 100 /* KeybindingWeight.EditorContrib */
            }
        });
    }
}
exports.DeleteWordRight = DeleteWordRight;
class DeleteInsideWord extends editorExtensions_1.EditorAction {
    constructor() {
        super({
            id: 'deleteInsideWord',
            precondition: editorContextKeys_1.EditorContextKeys.writable,
            label: nls.localize('deleteInsideWord', "Delete Word"),
            alias: 'Delete Word'
        });
    }
    run(accessor, editor, args) {
        if (!editor.hasModel()) {
            return;
        }
        const wordSeparators = (0, wordCharacterClassifier_1.getMapForWordSeparators)(editor.getOption(119 /* EditorOption.wordSeparators */));
        const model = editor.getModel();
        const selections = editor.getSelections();
        const commands = selections.map((sel) => {
            const deleteRange = cursorWordOperations_1.WordOperations.deleteInsideWord(wordSeparators, model, sel);
            return new replaceCommand_1.ReplaceCommand(deleteRange, '');
        });
        editor.pushUndoStop();
        editor.executeCommands(this.id, commands);
        editor.pushUndoStop();
    }
}
exports.DeleteInsideWord = DeleteInsideWord;
(0, editorExtensions_1.registerEditorCommand)(new CursorWordStartLeft());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordEndLeft());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordLeft());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordStartLeftSelect());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordEndLeftSelect());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordLeftSelect());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordStartRight());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordEndRight());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordRight());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordStartRightSelect());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordEndRightSelect());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordRightSelect());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordAccessibilityLeft());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordAccessibilityLeftSelect());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordAccessibilityRight());
(0, editorExtensions_1.registerEditorCommand)(new CursorWordAccessibilityRightSelect());
(0, editorExtensions_1.registerEditorCommand)(new DeleteWordStartLeft());
(0, editorExtensions_1.registerEditorCommand)(new DeleteWordEndLeft());
(0, editorExtensions_1.registerEditorCommand)(new DeleteWordLeft());
(0, editorExtensions_1.registerEditorCommand)(new DeleteWordStartRight());
(0, editorExtensions_1.registerEditorCommand)(new DeleteWordEndRight());
(0, editorExtensions_1.registerEditorCommand)(new DeleteWordRight());
(0, editorExtensions_1.registerEditorAction)(DeleteInsideWord);
//# sourceMappingURL=wordOperations.js.map