"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeActionKeybindingResolver = void 0;
const lazy_1 = require("../../../../base/common/lazy");
const codeAction_1 = require("./codeAction");
const types_1 = require("./types");
class CodeActionKeybindingResolver {
    constructor(keybindingService) {
        this.keybindingService = keybindingService;
    }
    getResolver() {
        // Lazy since we may not actually ever read the value
        const allCodeActionBindings = new lazy_1.Lazy(() => this.keybindingService.getKeybindings()
            .filter(item => CodeActionKeybindingResolver.codeActionCommands.indexOf(item.command) >= 0)
            .filter(item => item.resolvedKeybinding)
            .map((item) => {
            // Special case these commands since they come built-in with VS Code and don't use 'commandArgs'
            let commandArgs = item.commandArgs;
            if (item.command === codeAction_1.organizeImportsCommandId) {
                commandArgs = { kind: types_1.CodeActionKind.SourceOrganizeImports.value };
            }
            else if (item.command === codeAction_1.fixAllCommandId) {
                commandArgs = { kind: types_1.CodeActionKind.SourceFixAll.value };
            }
            return Object.assign({ resolvedKeybinding: item.resolvedKeybinding }, types_1.CodeActionCommandArgs.fromUser(commandArgs, {
                kind: types_1.CodeActionKind.None,
                apply: "never" /* CodeActionAutoApply.Never */
            }));
        }));
        return (action) => {
            if (action.kind) {
                const binding = this.bestKeybindingForCodeAction(action, allCodeActionBindings.getValue());
                return binding === null || binding === void 0 ? void 0 : binding.resolvedKeybinding;
            }
            return undefined;
        };
    }
    bestKeybindingForCodeAction(action, candidates) {
        if (!action.kind) {
            return undefined;
        }
        const kind = new types_1.CodeActionKind(action.kind);
        return candidates
            .filter(candidate => candidate.kind.contains(kind))
            .filter(candidate => {
            if (candidate.preferred) {
                // If the candidate keybinding only applies to preferred actions, the this action must also be preferred
                return action.isPreferred;
            }
            return true;
        })
            .reduceRight((currentBest, candidate) => {
            if (!currentBest) {
                return candidate;
            }
            // Select the more specific binding
            return currentBest.kind.contains(candidate.kind) ? candidate : currentBest;
        }, undefined);
    }
}
exports.CodeActionKeybindingResolver = CodeActionKeybindingResolver;
CodeActionKeybindingResolver.codeActionCommands = [
    codeAction_1.refactorCommandId,
    codeAction_1.codeActionCommandId,
    codeAction_1.sourceActionCommandId,
    codeAction_1.organizeImportsCommandId,
    codeAction_1.fixAllCommandId
];
//# sourceMappingURL=codeActionKeybindingResolver.js.map