"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoFixAction = exports.FixAllAction = exports.OrganizeImportsAction = exports.SourceAction = exports.RefactorPreview = exports.RefactorAction = exports.CodeActionCommand = exports.QuickFixAction = exports.applyCodeAction = exports.ApplyCodeActionReason = exports.CodeActionController = void 0;
const cancellation_1 = require("../../../../base/common/cancellation");
const lazy_1 = require("../../../../base/common/lazy");
const lifecycle_1 = require("../../../../base/common/lifecycle");
const strings_1 = require("../../../../base/common/strings");
const editorExtensions_1 = require("../../../browser/editorExtensions");
const bulkEditService_1 = require("../../../browser/services/bulkEditService");
const editorContextKeys_1 = require("../../../common/editorContextKeys");
const languageFeatures_1 = require("../../../common/services/languageFeatures");
const codeAction_1 = require("./codeAction");
const codeActionWidget_1 = require("./codeActionWidget");
const codeActionUi_1 = require("./codeActionUi");
const messageController_1 = require("../../message/browser/messageController");
const nls = require("../../../../nls");
const actions_1 = require("../../../../platform/actions/common/actions");
const commands_1 = require("../../../../platform/commands/common/commands");
const contextkey_1 = require("../../../../platform/contextkey/common/contextkey");
const instantiation_1 = require("../../../../platform/instantiation/common/instantiation");
const markers_1 = require("../../../../platform/markers/common/markers");
const notification_1 = require("../../../../platform/notification/common/notification");
const progress_1 = require("../../../../platform/progress/common/progress");
const telemetry_1 = require("../../../../platform/telemetry/common/telemetry");
const codeActionModel_1 = require("./codeActionModel");
const types_1 = require("./types");
function contextKeyForSupportedActions(kind) {
    return contextkey_1.ContextKeyExpr.regex(codeActionModel_1.SUPPORTED_CODE_ACTIONS.keys()[0], new RegExp('(\\s|^)' + (0, strings_1.escapeRegExpCharacters)(kind.value) + '\\b'));
}
function refactorTrigger(editor, userArgs, preview, codeActionFrom) {
    const args = types_1.CodeActionCommandArgs.fromUser(userArgs, {
        kind: types_1.CodeActionKind.Refactor,
        apply: "never" /* CodeActionAutoApply.Never */
    });
    return triggerCodeActionsForEditorSelection(editor, typeof (userArgs === null || userArgs === void 0 ? void 0 : userArgs.kind) === 'string'
        ? args.preferred
            ? nls.localize('editor.action.refactor.noneMessage.preferred.kind', "No preferred refactorings for '{0}' available", userArgs.kind)
            : nls.localize('editor.action.refactor.noneMessage.kind', "No refactorings for '{0}' available", userArgs.kind)
        : args.preferred
            ? nls.localize('editor.action.refactor.noneMessage.preferred', "No preferred refactorings available")
            : nls.localize('editor.action.refactor.noneMessage', "No refactorings available"), {
        include: types_1.CodeActionKind.Refactor.contains(args.kind) ? args.kind : types_1.CodeActionKind.None,
        onlyIncludePreferredActions: args.preferred
    }, args.apply, preview, codeActionFrom);
}
const argsSchema = {
    type: 'object',
    defaultSnippets: [{ body: { kind: '' } }],
    properties: {
        'kind': {
            type: 'string',
            description: nls.localize('args.schema.kind', "Kind of the code action to run."),
        },
        'apply': {
            type: 'string',
            description: nls.localize('args.schema.apply', "Controls when the returned actions are applied."),
            default: "ifSingle" /* CodeActionAutoApply.IfSingle */,
            enum: ["first" /* CodeActionAutoApply.First */, "ifSingle" /* CodeActionAutoApply.IfSingle */, "never" /* CodeActionAutoApply.Never */],
            enumDescriptions: [
                nls.localize('args.schema.apply.first', "Always apply the first returned code action."),
                nls.localize('args.schema.apply.ifSingle', "Apply the first returned code action if it is the only one."),
                nls.localize('args.schema.apply.never', "Do not apply the returned code actions."),
            ]
        },
        'preferred': {
            type: 'boolean',
            default: false,
            description: nls.localize('args.schema.preferred', "Controls if only preferred code actions should be returned."),
        }
    }
};
let CodeActionController = class CodeActionController extends lifecycle_1.Disposable {
    static get(editor) {
        return editor.getContribution(CodeActionController.ID);
    }
    constructor(editor, markerService, contextKeyService, progressService, _instantiationService, languageFeaturesService) {
        super();
        this._instantiationService = _instantiationService;
        this._editor = editor;
        this._model = this._register(new codeActionModel_1.CodeActionModel(this._editor, languageFeaturesService.codeActionProvider, markerService, contextKeyService, progressService));
        this._register(this._model.onDidChangeState(newState => this.update(newState)));
        this._ui = new lazy_1.Lazy(() => this._register(_instantiationService.createInstance(codeActionUi_1.CodeActionUi, editor, QuickFixAction.Id, AutoFixAction.Id, {
            applyCodeAction: (action, retrigger, preview) => __awaiter(this, void 0, void 0, function* () {
                try {
                    yield this._applyCodeAction(action, preview);
                }
                finally {
                    if (retrigger) {
                        this._trigger({ type: 2 /* CodeActionTriggerType.Auto */, triggerAction: types_1.CodeActionTriggerSource.QuickFix, filter: {} });
                    }
                }
            })
        })));
    }
    update(newState) {
        this._ui.getValue().update(newState);
    }
    showCodeActions(trigger, actions, at) {
        return this._ui.getValue().showCodeActionList(trigger, actions, at, { includeDisabledActions: false, fromLightbulb: false });
    }
    manualTriggerAtCurrentPosition(notAvailableMessage, triggerAction, filter, autoApply, preview) {
        var _a;
        if (!this._editor.hasModel()) {
            return;
        }
        (_a = messageController_1.MessageController.get(this._editor)) === null || _a === void 0 ? void 0 : _a.closeMessage();
        const triggerPosition = this._editor.getPosition();
        this._trigger({ type: 1 /* CodeActionTriggerType.Invoke */, triggerAction, filter, autoApply, context: { notAvailableMessage, position: triggerPosition }, preview });
    }
    _trigger(trigger) {
        return this._model.trigger(trigger);
    }
    _applyCodeAction(action, preview) {
        return this._instantiationService.invokeFunction(applyCodeAction, action, ApplyCodeActionReason.FromCodeActions, { preview, editor: this._editor });
    }
};
CodeActionController.ID = 'editor.contrib.codeActionController';
CodeActionController = __decorate([
    __param(1, markers_1.IMarkerService),
    __param(2, contextkey_1.IContextKeyService),
    __param(3, progress_1.IEditorProgressService),
    __param(4, instantiation_1.IInstantiationService),
    __param(5, languageFeatures_1.ILanguageFeaturesService)
], CodeActionController);
exports.CodeActionController = CodeActionController;
var ApplyCodeActionReason;
(function (ApplyCodeActionReason) {
    ApplyCodeActionReason["OnSave"] = "onSave";
    ApplyCodeActionReason["FromProblemsView"] = "fromProblemsView";
    ApplyCodeActionReason["FromCodeActions"] = "fromCodeActions";
})(ApplyCodeActionReason = exports.ApplyCodeActionReason || (exports.ApplyCodeActionReason = {}));
function applyCodeAction(accessor, item, codeActionReason, options) {
    return __awaiter(this, void 0, void 0, function* () {
        const bulkEditService = accessor.get(bulkEditService_1.IBulkEditService);
        const commandService = accessor.get(commands_1.ICommandService);
        const telemetryService = accessor.get(telemetry_1.ITelemetryService);
        const notificationService = accessor.get(notification_1.INotificationService);
        telemetryService.publicLog2('codeAction.applyCodeAction', {
            codeActionTitle: item.action.title,
            codeActionKind: item.action.kind,
            codeActionIsPreferred: !!item.action.isPreferred,
            reason: codeActionReason,
        });
        yield item.resolve(cancellation_1.CancellationToken.None);
        if (item.action.edit) {
            yield bulkEditService.apply(item.action.edit, {
                editor: options === null || options === void 0 ? void 0 : options.editor,
                label: item.action.title,
                quotableLabel: item.action.title,
                code: 'undoredo.codeAction',
                respectAutoSaveConfig: true,
                showPreview: options === null || options === void 0 ? void 0 : options.preview,
            });
        }
        if (item.action.command) {
            try {
                yield commandService.executeCommand(item.action.command.id, ...(item.action.command.arguments || []));
            }
            catch (err) {
                const message = asMessage(err);
                notificationService.error(typeof message === 'string'
                    ? message
                    : nls.localize('applyCodeActionFailed', "An unknown error occurred while applying the code action"));
            }
        }
    });
}
exports.applyCodeAction = applyCodeAction;
function asMessage(err) {
    if (typeof err === 'string') {
        return err;
    }
    else if (err instanceof Error && typeof err.message === 'string') {
        return err.message;
    }
    else {
        return undefined;
    }
}
function triggerCodeActionsForEditorSelection(editor, notAvailableMessage, filter, autoApply, preview = false, triggerAction = types_1.CodeActionTriggerSource.Default) {
    if (editor.hasModel()) {
        const controller = CodeActionController.get(editor);
        controller === null || controller === void 0 ? void 0 : controller.manualTriggerAtCurrentPosition(notAvailableMessage, triggerAction, filter, autoApply, preview);
    }
}
class QuickFixAction extends editorExtensions_1.EditorAction {
    constructor() {
        super({
            id: QuickFixAction.Id,
            label: nls.localize('quickfix.trigger.label', "Quick Fix..."),
            alias: 'Quick Fix...',
            precondition: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, editorContextKeys_1.EditorContextKeys.hasCodeActionsProvider),
            kbOpts: {
                kbExpr: editorContextKeys_1.EditorContextKeys.editorTextFocus,
                primary: 2048 /* KeyMod.CtrlCmd */ | 84 /* KeyCode.Period */,
                weight: 100 /* KeybindingWeight.EditorContrib */
            }
        });
    }
    run(_accessor, editor) {
        return triggerCodeActionsForEditorSelection(editor, nls.localize('editor.action.quickFix.noneMessage', "No code actions available"), undefined, undefined, false, types_1.CodeActionTriggerSource.QuickFix);
    }
}
exports.QuickFixAction = QuickFixAction;
QuickFixAction.Id = 'editor.action.quickFix';
class CodeActionCommand extends editorExtensions_1.EditorCommand {
    constructor() {
        super({
            id: codeAction_1.codeActionCommandId,
            precondition: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, editorContextKeys_1.EditorContextKeys.hasCodeActionsProvider),
            description: {
                description: 'Trigger a code action',
                args: [{ name: 'args', schema: argsSchema, }]
            }
        });
    }
    runEditorCommand(_accessor, editor, userArgs) {
        const args = types_1.CodeActionCommandArgs.fromUser(userArgs, {
            kind: types_1.CodeActionKind.Empty,
            apply: "ifSingle" /* CodeActionAutoApply.IfSingle */,
        });
        return triggerCodeActionsForEditorSelection(editor, typeof (userArgs === null || userArgs === void 0 ? void 0 : userArgs.kind) === 'string'
            ? args.preferred
                ? nls.localize('editor.action.codeAction.noneMessage.preferred.kind', "No preferred code actions for '{0}' available", userArgs.kind)
                : nls.localize('editor.action.codeAction.noneMessage.kind', "No code actions for '{0}' available", userArgs.kind)
            : args.preferred
                ? nls.localize('editor.action.codeAction.noneMessage.preferred', "No preferred code actions available")
                : nls.localize('editor.action.codeAction.noneMessage', "No code actions available"), {
            include: args.kind,
            includeSourceActions: true,
            onlyIncludePreferredActions: args.preferred,
        }, args.apply);
    }
}
exports.CodeActionCommand = CodeActionCommand;
class RefactorAction extends editorExtensions_1.EditorAction {
    constructor() {
        super({
            id: codeAction_1.refactorCommandId,
            label: nls.localize('refactor.label', "Refactor..."),
            alias: 'Refactor...',
            precondition: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, editorContextKeys_1.EditorContextKeys.hasCodeActionsProvider),
            kbOpts: {
                kbExpr: editorContextKeys_1.EditorContextKeys.editorTextFocus,
                primary: 2048 /* KeyMod.CtrlCmd */ | 1024 /* KeyMod.Shift */ | 48 /* KeyCode.KeyR */,
                mac: {
                    primary: 256 /* KeyMod.WinCtrl */ | 1024 /* KeyMod.Shift */ | 48 /* KeyCode.KeyR */
                },
                weight: 100 /* KeybindingWeight.EditorContrib */
            },
            contextMenuOpts: {
                group: '1_modification',
                order: 2,
                when: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, contextKeyForSupportedActions(types_1.CodeActionKind.Refactor)),
            },
            description: {
                description: 'Refactor...',
                args: [{ name: 'args', schema: argsSchema }]
            }
        });
    }
    run(_accessor, editor, userArgs) {
        return refactorTrigger(editor, userArgs, false, types_1.CodeActionTriggerSource.Refactor);
    }
}
exports.RefactorAction = RefactorAction;
class RefactorPreview extends editorExtensions_1.EditorAction {
    constructor() {
        super({
            id: codeAction_1.refactorPreviewCommandId,
            label: nls.localize('refactor.preview.label', "Refactor with Preview..."),
            alias: 'Refactor Preview...',
            precondition: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, editorContextKeys_1.EditorContextKeys.hasCodeActionsProvider),
            description: {
                description: 'Refactor Preview...',
                args: [{ name: 'args', schema: argsSchema }]
            }
        });
    }
    run(_accessor, editor, userArgs) {
        return refactorTrigger(editor, userArgs, true, types_1.CodeActionTriggerSource.RefactorPreview);
    }
}
exports.RefactorPreview = RefactorPreview;
class SourceAction extends editorExtensions_1.EditorAction {
    constructor() {
        super({
            id: codeAction_1.sourceActionCommandId,
            label: nls.localize('source.label', "Source Action..."),
            alias: 'Source Action...',
            precondition: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, editorContextKeys_1.EditorContextKeys.hasCodeActionsProvider),
            contextMenuOpts: {
                group: '1_modification',
                order: 2.1,
                when: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, contextKeyForSupportedActions(types_1.CodeActionKind.Source)),
            },
            description: {
                description: 'Source Action...',
                args: [{ name: 'args', schema: argsSchema }]
            }
        });
    }
    run(_accessor, editor, userArgs) {
        const args = types_1.CodeActionCommandArgs.fromUser(userArgs, {
            kind: types_1.CodeActionKind.Source,
            apply: "never" /* CodeActionAutoApply.Never */
        });
        return triggerCodeActionsForEditorSelection(editor, typeof (userArgs === null || userArgs === void 0 ? void 0 : userArgs.kind) === 'string'
            ? args.preferred
                ? nls.localize('editor.action.source.noneMessage.preferred.kind', "No preferred source actions for '{0}' available", userArgs.kind)
                : nls.localize('editor.action.source.noneMessage.kind', "No source actions for '{0}' available", userArgs.kind)
            : args.preferred
                ? nls.localize('editor.action.source.noneMessage.preferred', "No preferred source actions available")
                : nls.localize('editor.action.source.noneMessage', "No source actions available"), {
            include: types_1.CodeActionKind.Source.contains(args.kind) ? args.kind : types_1.CodeActionKind.None,
            includeSourceActions: true,
            onlyIncludePreferredActions: args.preferred,
        }, args.apply, undefined, types_1.CodeActionTriggerSource.SourceAction);
    }
}
exports.SourceAction = SourceAction;
class OrganizeImportsAction extends editorExtensions_1.EditorAction {
    constructor() {
        super({
            id: codeAction_1.organizeImportsCommandId,
            label: nls.localize('organizeImports.label', "Organize Imports"),
            alias: 'Organize Imports',
            precondition: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, contextKeyForSupportedActions(types_1.CodeActionKind.SourceOrganizeImports)),
            kbOpts: {
                kbExpr: editorContextKeys_1.EditorContextKeys.editorTextFocus,
                primary: 1024 /* KeyMod.Shift */ | 512 /* KeyMod.Alt */ | 45 /* KeyCode.KeyO */,
                weight: 100 /* KeybindingWeight.EditorContrib */
            },
        });
    }
    run(_accessor, editor) {
        return triggerCodeActionsForEditorSelection(editor, nls.localize('editor.action.organize.noneMessage', "No organize imports action available"), { include: types_1.CodeActionKind.SourceOrganizeImports, includeSourceActions: true }, "ifSingle" /* CodeActionAutoApply.IfSingle */, undefined, types_1.CodeActionTriggerSource.OrganizeImports);
    }
}
exports.OrganizeImportsAction = OrganizeImportsAction;
class FixAllAction extends editorExtensions_1.EditorAction {
    constructor() {
        super({
            id: codeAction_1.fixAllCommandId,
            label: nls.localize('fixAll.label', "Fix All"),
            alias: 'Fix All',
            precondition: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, contextKeyForSupportedActions(types_1.CodeActionKind.SourceFixAll))
        });
    }
    run(_accessor, editor) {
        return triggerCodeActionsForEditorSelection(editor, nls.localize('fixAll.noneMessage', "No fix all action available"), { include: types_1.CodeActionKind.SourceFixAll, includeSourceActions: true }, "ifSingle" /* CodeActionAutoApply.IfSingle */, undefined, types_1.CodeActionTriggerSource.FixAll);
    }
}
exports.FixAllAction = FixAllAction;
class AutoFixAction extends editorExtensions_1.EditorAction {
    constructor() {
        super({
            id: AutoFixAction.Id,
            label: nls.localize('autoFix.label', "Auto Fix..."),
            alias: 'Auto Fix...',
            precondition: contextkey_1.ContextKeyExpr.and(editorContextKeys_1.EditorContextKeys.writable, contextKeyForSupportedActions(types_1.CodeActionKind.QuickFix)),
            kbOpts: {
                kbExpr: editorContextKeys_1.EditorContextKeys.editorTextFocus,
                primary: 512 /* KeyMod.Alt */ | 1024 /* KeyMod.Shift */ | 84 /* KeyCode.Period */,
                mac: {
                    primary: 2048 /* KeyMod.CtrlCmd */ | 512 /* KeyMod.Alt */ | 84 /* KeyCode.Period */
                },
                weight: 100 /* KeybindingWeight.EditorContrib */
            }
        });
    }
    run(_accessor, editor) {
        return triggerCodeActionsForEditorSelection(editor, nls.localize('editor.action.autoFix.noneMessage', "No auto fixes available"), {
            include: types_1.CodeActionKind.QuickFix,
            onlyIncludePreferredActions: true
        }, "ifSingle" /* CodeActionAutoApply.IfSingle */, undefined, types_1.CodeActionTriggerSource.AutoFix);
    }
}
exports.AutoFixAction = AutoFixAction;
AutoFixAction.Id = 'editor.action.autoFix';
const weight = 100 /* KeybindingWeight.EditorContrib */ + 90;
(0, actions_1.registerAction2)(class extends actions_1.Action2 {
    constructor() {
        super({
            id: 'hideCodeActionWidget',
            title: {
                value: nls.localize('hideCodeActionWidget.title', "Hide code action widget"),
                original: 'Hide code action widget'
            },
            precondition: codeActionWidget_1.Context.Visible,
            keybinding: {
                weight: weight,
                primary: 9 /* KeyCode.Escape */,
                secondary: [1024 /* KeyMod.Shift */ | 9 /* KeyCode.Escape */]
            },
        });
    }
    run() {
        var _a;
        (_a = codeActionWidget_1.CodeActionWidget.INSTANCE) === null || _a === void 0 ? void 0 : _a.hide();
    }
});
(0, actions_1.registerAction2)(class extends actions_1.Action2 {
    constructor() {
        super({
            id: 'selectPrevCodeAction',
            title: {
                value: nls.localize('selectPrevCodeAction.title', "Select previous code action"),
                original: 'Select previous code action'
            },
            precondition: codeActionWidget_1.Context.Visible,
            keybinding: {
                weight: weight + 100000,
                primary: 16 /* KeyCode.UpArrow */,
                secondary: [2048 /* KeyMod.CtrlCmd */ | 16 /* KeyCode.UpArrow */],
                mac: { primary: 16 /* KeyCode.UpArrow */, secondary: [2048 /* KeyMod.CtrlCmd */ | 16 /* KeyCode.UpArrow */, 256 /* KeyMod.WinCtrl */ | 46 /* KeyCode.KeyP */] },
            }
        });
    }
    run() {
        var _a;
        (_a = codeActionWidget_1.CodeActionWidget.INSTANCE) === null || _a === void 0 ? void 0 : _a.focusPrevious();
    }
});
(0, actions_1.registerAction2)(class extends actions_1.Action2 {
    constructor() {
        super({
            id: 'selectNextCodeAction',
            title: {
                value: nls.localize('selectNextCodeAction.title', "Select next code action"),
                original: 'Select next code action'
            },
            precondition: codeActionWidget_1.Context.Visible,
            keybinding: {
                weight: weight + 100000,
                primary: 18 /* KeyCode.DownArrow */,
                secondary: [2048 /* KeyMod.CtrlCmd */ | 18 /* KeyCode.DownArrow */],
                mac: { primary: 18 /* KeyCode.DownArrow */, secondary: [2048 /* KeyMod.CtrlCmd */ | 18 /* KeyCode.DownArrow */, 256 /* KeyMod.WinCtrl */ | 44 /* KeyCode.KeyN */] }
            }
        });
    }
    run() {
        var _a;
        (_a = codeActionWidget_1.CodeActionWidget.INSTANCE) === null || _a === void 0 ? void 0 : _a.focusNext();
    }
});
(0, actions_1.registerAction2)(class extends actions_1.Action2 {
    constructor() {
        super({
            id: codeActionWidget_1.acceptSelectedCodeActionCommand,
            title: {
                value: nls.localize('acceptSelected.title', "Accept selected code action"),
                original: 'Accept selected code action'
            },
            precondition: codeActionWidget_1.Context.Visible,
            keybinding: {
                weight: weight + 100000,
                primary: 3 /* KeyCode.Enter */,
                secondary: [2048 /* KeyMod.CtrlCmd */ | 84 /* KeyCode.Period */],
            }
        });
    }
    run() {
        var _a;
        (_a = codeActionWidget_1.CodeActionWidget.INSTANCE) === null || _a === void 0 ? void 0 : _a.acceptSelected();
    }
});
(0, actions_1.registerAction2)(class extends actions_1.Action2 {
    constructor() {
        super({
            id: codeActionWidget_1.previewSelectedCodeActionCommand,
            title: {
                value: nls.localize('previewSelected.title', "Preview selected code action"),
                original: 'Preview selected code action'
            },
            precondition: codeActionWidget_1.Context.Visible,
            keybinding: {
                weight: weight + 100000,
                primary: 2048 /* KeyMod.CtrlCmd */ | 3 /* KeyCode.Enter */,
            }
        });
    }
    run() {
        var _a;
        (_a = codeActionWidget_1.CodeActionWidget.INSTANCE) === null || _a === void 0 ? void 0 : _a.acceptSelected({ preview: true });
    }
});
//# sourceMappingURL=codeActionCommands.js.map