/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 *
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.jdt.jeview.views;

import java.util.ArrayList;
import java.util.Objects;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;

import org.eclipse.jdt.core.IJarEntryResource;
import org.eclipse.jdt.core.IJavaElement;


public class JEJarEntryResource extends JEAttribute {

	private final JEAttribute fParent; // can be null
	private final String fName; // can be null
	private IJarEntryResource fJarEntryResource;

	JEJarEntryResource(JEAttribute parent, String name, IJarEntryResource jarEntryResource) {
		Assert.isNotNull(jarEntryResource);
		fParent= parent;
		fName= name;
		fJarEntryResource= jarEntryResource;
	}

	@Override
	public JEAttribute getParent() {
		return fParent;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null || !obj.getClass().equals(getClass())) {
			return false;
		}

		JEJarEntryResource other= (JEJarEntryResource) obj;
		if (!Objects.equals(fParent, other.fParent)) {
			return false;
		}

		if (!Objects.equals(fName, other.fName)) {
			return false;
		}

		return true;
	}

	@Override
	public int hashCode() {
		return (fParent != null ? fParent.hashCode() : 0)
				+ (fName != null ? fName.hashCode() : 0)
				+ fJarEntryResource.hashCode();
	}

	@Override
	public Object getWrappedObject() {
		return fJarEntryResource;
	}

	public IJarEntryResource getJarEntryResource() {
		return fJarEntryResource;
	}

	@Override
	public JEAttribute[] getChildren() {
		ArrayList<JEAttribute> result= new ArrayList<>();

		Object parent= fJarEntryResource.getParent();
		if (parent instanceof IJarEntryResource)
			result.add(new JEJarEntryResource(this, "PARENT", (IJarEntryResource) parent));
		else
			result.add(new JavaElement(this, "PARENT", (IJavaElement) parent));

		result.add(new JavaElement(this, "PACKAGE FRAGMENT ROOT", fJarEntryResource.getPackageFragmentRoot()));

		result.add(new JavaElementChildrenProperty(this, "CHILDREN") {
			@Override protected JEAttribute[] computeChildren() throws CoreException {
				IJarEntryResource[] jarEntryResources= getJarEntryResource().getChildren();
				JEAttribute[] children= new JEAttribute[jarEntryResources.length];
				for (int i= 0; i < jarEntryResources.length; i++) {
					children[i]= new JEJarEntryResource(this, null, jarEntryResources[i]);
				}
				return children;
			}
		});
		return result.toArray(new JEAttribute[result.size()]);
	}

	@Override
	public String getLabel() {
		StringBuilder sb= new StringBuilder();
		if (fName != null)
			sb.append(fName).append(": ");
		String classname= fJarEntryResource.getClass().getName();
		sb.append(classname.substring(classname.lastIndexOf('.') + 1)).append(": ");
		sb.append(fJarEntryResource.getName());
		return sb.toString();
	}

}
