/*******************************************************************************
 * Copyright (c) 2000, 2017 IBM Corporation and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.dltk.ui.preferences;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.dltk.compiler.task.TaskTagUtils;
import org.eclipse.dltk.compiler.task.TodoTask;
import org.eclipse.dltk.internal.ui.wizards.dialogfields.ComboDialogField;
import org.eclipse.dltk.internal.ui.wizards.dialogfields.DialogField;
import org.eclipse.dltk.internal.ui.wizards.dialogfields.IDialogFieldListener;
import org.eclipse.dltk.internal.ui.wizards.dialogfields.LayoutUtil;
import org.eclipse.dltk.internal.ui.wizards.dialogfields.StringDialogField;
import org.eclipse.dltk.ui.dialogs.StatusInfo;
import org.eclipse.jface.dialogs.StatusDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;

/**
 * Dialog to enter a new task tag
 */
public class TodoTaskInputDialog extends StatusDialog {

	private class CompilerTodoTaskInputAdapter implements IDialogFieldListener {
		@Override
		public void dialogFieldChanged(DialogField field) {
			doValidation();
		}
	}

	private StringDialogField fNameDialogField;
	private ComboDialogField fPriorityDialogField;

	private List fExistingNames;

	public TodoTaskInputDialog(Shell parent, TodoTask task, List existingEntries) {
		super(parent);

		fExistingNames = new ArrayList(existingEntries.size());
		for (int i = 0; i < existingEntries.size(); i++) {
			TodoTask curr = (TodoTask) existingEntries.get(i);
			if (!curr.equals(task)) {
				fExistingNames.add(curr.name);
			}
		}

		if (task == null) {
			setTitle(PreferencesMessages.TodoTaskInputDialog_new_title);
		} else {
			setTitle(PreferencesMessages.TodoTaskInputDialog_edit_title);
		}

		CompilerTodoTaskInputAdapter adapter = new CompilerTodoTaskInputAdapter();

		fNameDialogField = new StringDialogField();
		fNameDialogField
				.setLabelText(PreferencesMessages.TodoTaskInputDialog_name_label);
		fNameDialogField.setDialogFieldListener(adapter);

		fNameDialogField.setText((task != null) ? task.name : ""); //$NON-NLS-1$

		String[] items = new String[] {
				PreferencesMessages.TodoTaskInputDialog_priority_high,
				PreferencesMessages.TodoTaskInputDialog_priority_normal,
				PreferencesMessages.TodoTaskInputDialog_priority_low };

		fPriorityDialogField = new ComboDialogField(SWT.READ_ONLY);
		fPriorityDialogField
				.setLabelText(PreferencesMessages.TodoTaskInputDialog_priority_label);
		fPriorityDialogField.setItems(items);
		if (task != null) {
			if (TodoTask.PRIORITY_HIGH.equals(task.priority)) {
				fPriorityDialogField.selectItem(0);
			} else if (TodoTask.PRIORITY_NORMAL.equals(task.priority)) {
				fPriorityDialogField.selectItem(1);
			} else {
				fPriorityDialogField.selectItem(2);
			}
		} else {
			fPriorityDialogField.selectItem(1);
		}
	}

	public TodoTask getResult() {
		TodoTask task = new TodoTask();
		task.name = fNameDialogField.getText().trim();
		switch (fPriorityDialogField.getSelectionIndex()) {
		case 0:
			task.priority = TodoTask.PRIORITY_HIGH;
			break;
		case 1:
			task.priority = TodoTask.PRIORITY_NORMAL;
			break;
		default:
			task.priority = TodoTask.PRIORITY_LOW;
			break;
		}
		return task;
	}

	@Override
	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);

		Composite inner = new Composite(composite, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		layout.numColumns = 2;
		inner.setLayout(layout);

		fNameDialogField.doFillIntoGrid(inner, 2);
		fPriorityDialogField.doFillIntoGrid(inner, 2);

		LayoutUtil.setHorizontalGrabbing(fNameDialogField.getTextControl(null));
		LayoutUtil.setWidthHint(fNameDialogField.getTextControl(null),
				convertWidthInCharsToPixels(45));

		fNameDialogField.postSetFocusOnDialogField(parent.getDisplay());

		applyDialogFont(composite);

		return composite;
	}

	private void doValidation() {
		StatusInfo status = new StatusInfo();
		String newText = fNameDialogField.getText();
		if (newText.length() == 0) {
			status
					.setError(PreferencesMessages.TodoTaskInputDialog_error_enterName);
		} else {
			if (!TaskTagUtils.isValidName(newText)) {
				status
						.setError(PreferencesMessages.TodoTaskInputDialog_error_comma);
			} else if (fExistingNames.contains(newText)) {
				status
						.setError(PreferencesMessages.TodoTaskInputDialog_error_entryExists);
			} else if (Character.isWhitespace(newText.charAt(0))
					|| Character.isWhitespace(newText
							.charAt(newText.length() - 1))) {
				status
						.setError(PreferencesMessages.TodoTaskInputDialog_error_noSpace);
			}
		}
		updateStatus(status);
	}

	@Override
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
	}
}
