/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * A string value used as a snippet is a template which allows to insert text
 * and to control the editor cursor when insertion happens.
 * <p>
 * A snippet can define tab stops and placeholders with `$1`, `$2`
 * and `${3:foo}`. `$0` defines the final tab stop, it defaults to
 * the end of the snippet. Variables are defined with `$name` and
 * `${name:default value}`.
 */
@ProtocolDraft
@ProtocolSince("3.18.0")
@SuppressWarnings("all")
public class StringValue {
  /**
   * The kind of the string value.
   * <p>
   * See {@link StringValueKind} for allowed values.
   */
  @NonNull
  private String kind;

  /**
   * The string value.
   */
  @NonNull
  private String value;

  public StringValue() {
  }

  public StringValue(@NonNull final String kind, @NonNull final String value) {
    this.kind = Preconditions.<String>checkNotNull(kind, "kind");
    this.value = Preconditions.<String>checkNotNull(value, "value");
  }

  /**
   * The kind of the string value.
   * <p>
   * See {@link StringValueKind} for allowed values.
   */
  @NonNull
  public String getKind() {
    return this.kind;
  }

  /**
   * The kind of the string value.
   * <p>
   * See {@link StringValueKind} for allowed values.
   */
  public void setKind(@NonNull final String kind) {
    this.kind = Preconditions.checkNotNull(kind, "kind");
  }

  /**
   * The string value.
   */
  @NonNull
  public String getValue() {
    return this.value;
  }

  /**
   * The string value.
   */
  public void setValue(@NonNull final String value) {
    this.value = Preconditions.checkNotNull(value, "value");
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("kind", this.kind);
    b.add("value", this.value);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    StringValue other = (StringValue) obj;
    if (this.kind == null) {
      if (other.kind != null)
        return false;
    } else if (!this.kind.equals(other.kind))
      return false;
    if (this.value == null) {
      if (other.value != null)
        return false;
    } else if (!this.value.equals(other.value))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.kind== null) ? 0 : this.kind.hashCode());
    return prime * result + ((this.value== null) ? 0 : this.value.hashCode());
  }
}
