/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import org.eclipse.lsp4j.jsonrpc.ProtocolDraft;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * An interactive text edit.
 */
@ProtocolDraft
@ProtocolSince("3.18.0")
@SuppressWarnings("all")
public class SnippetTextEdit {
  /**
   * The range of the text document to be manipulated.
   */
  @NonNull
  private Range range;

  /**
   * The snippet to be inserted.
   */
  @NonNull
  private StringValue snippet;

  /**
   * The actual identifier of the snippet edit.
   */
  private String annotationId;

  public SnippetTextEdit() {
  }

  public SnippetTextEdit(@NonNull final Range range, @NonNull final StringValue snippet) {
    this.range = Preconditions.<Range>checkNotNull(range, "range");
    this.snippet = Preconditions.<StringValue>checkNotNull(snippet, "snippet");
  }

  public SnippetTextEdit(@NonNull final Range range, @NonNull final StringValue snippet, final String annotationId) {
    this(range, snippet);
    this.annotationId = annotationId;
  }

  /**
   * The range of the text document to be manipulated.
   */
  @NonNull
  public Range getRange() {
    return this.range;
  }

  /**
   * The range of the text document to be manipulated.
   */
  public void setRange(@NonNull final Range range) {
    this.range = Preconditions.checkNotNull(range, "range");
  }

  /**
   * The snippet to be inserted.
   */
  @NonNull
  public StringValue getSnippet() {
    return this.snippet;
  }

  /**
   * The snippet to be inserted.
   */
  public void setSnippet(@NonNull final StringValue snippet) {
    this.snippet = Preconditions.checkNotNull(snippet, "snippet");
  }

  /**
   * The actual identifier of the snippet edit.
   */
  public String getAnnotationId() {
    return this.annotationId;
  }

  /**
   * The actual identifier of the snippet edit.
   */
  public void setAnnotationId(final String annotationId) {
    this.annotationId = annotationId;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("range", this.range);
    b.add("snippet", this.snippet);
    b.add("annotationId", this.annotationId);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    SnippetTextEdit other = (SnippetTextEdit) obj;
    if (this.range == null) {
      if (other.range != null)
        return false;
    } else if (!this.range.equals(other.range))
      return false;
    if (this.snippet == null) {
      if (other.snippet != null)
        return false;
    } else if (!this.snippet.equals(other.snippet))
      return false;
    if (this.annotationId == null) {
      if (other.annotationId != null)
        return false;
    } else if (!this.annotationId.equals(other.annotationId))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.range== null) ? 0 : this.range.hashCode());
    result = prime * result + ((this.snippet== null) ? 0 : this.snippet.hashCode());
    return prime * result + ((this.annotationId== null) ? 0 : this.annotationId.hashCode());
  }
}
