/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.scheduler.product;

import static org.eclipse.lsat.scheduler.product.ProductUtil.createProductChanges;
import static org.eclipse.lsat.scheduler.product.ProductUtil.getProductInstanceForSlot;
import static org.eclipse.lsat.scheduler.product.ProductUtil.getProductInstanceThatHasNotPassedSlot;
import static org.eclipse.lsat.scheduler.product.ProductUtil.getProductLocationsString;
import static org.eclipse.lsat.scheduler.product.ProductUtil.logProductFlow;
import static org.eclipse.lsat.scheduler.product.ProductUtil.logProductLocations;
import static org.slf4j.LoggerFactory.getLogger;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.eclipse.lsat.common.scheduler.graph.Task;
import org.eclipse.lsat.common.scheduler.schedule.Schedule;
import org.eclipse.lsat.product.productdata.EntryEvent;
import org.eclipse.lsat.product.productdata.ExitEvent;
import org.eclipse.lsat.product.productdata.ProductInstance;
import org.eclipse.lsat.product.productdata.TransferEvent;
import org.eclipse.lsat.product.productdata.TransferInformation;
import org.slf4j.Logger;

import product.TransferType;

/**
 * Takes in a Schedule object and extracts all its product changes, e.g. product life cycles and property assignments.
 * These traces are collected and stored in data classes which are returned.
 */
public class ProductTransformer {
    private static final Logger LOGGER = getLogger(ProductTransformer.class);

    /**
     * Private constructor to prevent instantiation of utility class.
     */
    private ProductTransformer() {
        // no instances allowed
    }

    /**
     * Generates a list of product change information in the schedule.
     *
     * @param schedule The schedule containing product changes
     * @param debug Whether to output debug information
     * @return List of product instances
     */
    public static List<ProductInstance> getProductTracesFromSchedule(Schedule<Task> schedule, boolean debug) {
        var productChanges = createProductChanges(schedule);
        var allProducts = new ArrayList<ProductInstance>();
        // a product is owned by exactly one product owner we keep the latest product change
        var activeProducts = new LinkedHashMap<ProductInstance, ProductChange>();

        try {
            processProductChanges(productChanges, allProducts, activeProducts, debug);
        } catch (Exception e) {
            logProductState(activeProducts, allProducts);
            logErrorState(activeProducts, allProducts);
            throw e;
        }

        logProductState(activeProducts, allProducts);
        return allProducts;
    }

    /**
     * Process all product changes in the queue.
     */
    private static void processProductChanges(List<ProductChange> productChanges, List<ProductInstance> allProducts,
            Map<ProductInstance, ProductChange> activeProducts, boolean debug)
    {
        while (!productChanges.isEmpty()) {
            var productChange = productChanges.removeFirst();

            logProductChange(productChange, debug);

            processProductChange(productChange, productChanges, allProducts, activeProducts, debug);

            logProductStateIfNeeded(productChange, activeProducts, allProducts, debug);
        }
    }

    /**
     * Process a single product change based on its type.
     */
    private static void processProductChange(ProductChange productChange, List<ProductChange> productChanges,
            List<ProductInstance> allProducts, Map<ProductInstance, ProductChange> activeProducts, boolean debug)
    {
        if (productChange.isEntry()) {
            handleEntryEvent(productChange, allProducts, activeProducts);
        } else if (productChange.isTransfer()) {
            handleTransferEvent(productChange, productChanges, activeProducts, debug);
        } else if (productChange.isExit()) {
            handleExitEvent(productChange, activeProducts);
        } else {
            handlePropertyChange(productChange, activeProducts);
        }
    }

    /**
     * Handle a product load event.
     */
    private static void handleEntryEvent(ProductChange productChange, List<ProductInstance> allProducts,
            Map<ProductInstance, ProductChange> activeProducts)
    {
        var productInstance = startLifeCycle(Integer.toString(allProducts.size()), productChange);
        allProducts.add(productInstance);
        activeProducts.put(productInstance, productChange);
    }

    /**
     * Handle a product transfer event.
     */
    private static void handleTransferEvent(ProductChange productChange, List<ProductChange> productChanges,
            Map<ProductInstance, ProductChange> activeProducts, boolean debug)
    {
        var matchingContextOpt = findMatchingProductChange(productChange, productChanges);

        if (matchingContextOpt.isEmpty()) {
            throwMatchingContextNotFound(productChange, activeProducts);
        }

        var matchingContext = matchingContextOpt.get();
        productChanges.remove(matchingContext);

        if (debug) {
            LOGGER.info("Match: " + matchingContext);
        }

        transferLifeCycle(activeProducts, productChange, matchingContext);
    }

    /**
     * Handle a product unload event.
     */
    private static void handleExitEvent(ProductChange productChange,
            Map<ProductInstance, ProductChange> activeProducts)
    {
        endLifeCycle(activeProducts, productChange);
    }

    /**
     * Handle a product property change event.
     */
    private static void handlePropertyChange(ProductChange productChange,
            Map<ProductInstance, ProductChange> activeProducts)
    {
        propertyChange(activeProducts, productChange);
    }

    /**
     * Find a matching product change for a transfer operation.
     */
    private static Optional<ProductChange> findMatchingProductChange(ProductChange productChange,
            List<ProductChange> productChanges)
    {
        return productChanges.stream().filter(productChange::isMatch).findFirst();
    }

    /**
     * Throw an exception when a matching context is not found.
     */
    private static void throwMatchingContextNotFound(ProductChange productChange,
            Map<ProductInstance, ProductChange> activeProducts)
    {
        var action = productChange.getTaskData().getId();
        var activity = productChange.getProductOwner().getAction().getGraph().getName();

        var errorMessage = """
                Action: '%s.%s' - Cannot find a match for %s
                """.formatted(activity, action, productChange);

        throw new IllegalArgumentException(errorMessage);
    }

    /**
     * Log product change information if debug is enabled.
     */
    private static void logProductChange(ProductChange productChange, boolean debug) {
        if (debug && productChange.getTransferType() != TransferType.UNKNOWN) {
            LOGGER.info("ProductChange: " + productChange);
        }
    }

    /**
     * Log product state information if debug is enabled.
     */
    private static void logProductStateIfNeeded(ProductChange productChange,
            Map<ProductInstance, ProductChange> activeProducts, List<ProductInstance> allProducts, boolean debug)
    {
        if (debug && productChange.getTransferType() != TransferType.UNKNOWN) {
            logProductState(activeProducts, allProducts);
        }
    }

    /**
     * Log the current state of products.
     */
    private static void logProductState(Map<ProductInstance, ProductChange> activeProducts,
            List<ProductInstance> allProducts)
    {
        logProductLocations(activeProducts);
        logProductFlow(allProducts);
    }

    /**
     * Log error state information when an exception occurs.
     */
    private static void logErrorState(LinkedHashMap<ProductInstance, ProductChange> activeProducts,
            List<ProductInstance> allProducts)
    {
        LOGGER.error("Error occurred during product transformation. Current state:");
        logProductState(activeProducts, allProducts);
    }

    /**
     * Start a product lifecycle.
     */
    private static ProductInstance startLifeCycle(String productId, ProductChange prodChange) {
        var locationInformation = prodChange.createLocationInformation();
        var createEvent = new EntryEvent(locationInformation);

        // Converts the ecore Product model to the dataclass ProductProperties
        var propertyList = prodChange.getChangedProperties();
        // set the product id
        prodChange.initProduct(productId);

        // Creates the new product dataclass, adds it to the list.
        return new ProductInstance(productId, createEvent, propertyList);
    }

    /**
     * Handle a product transfer between locations.
     */
    private static void transferLifeCycle(Map<ProductInstance, ProductChange> activeProducts, ProductChange prodChange,
            ProductChange matchingProductChange)
    {
        var all = List.of(prodChange, matchingProductChange);
        // Assumes every task is of type PeripheralActionTask and every action is of type TransferAction
        var inProducts = all.stream().filter(pd -> pd.getTransferType() == TransferType.IN).toList();
        var outProducts = all.stream().filter(pd -> pd.getTransferType() == TransferType.OUT).toList();

        validateTransferActions(inProducts, outProducts);

        for (var inProductChange: inProducts) {
            var inPropertyList = inProductChange.getChangedProperties();
            var inLocation = inProductChange.createLocationInformation();

            for (var outProductChange: outProducts) {
                var outPropertyList = outProductChange.getChangedProperties();
                var outLocation = outProductChange.createLocationInformation();

                var productInstance = getProductInstanceThatHasNotPassedSlot(activeProducts,
                        outProductChange.getProductOwner(), outProductChange.getSlot());

                if (productInstance == null) {
                    throw new IllegalStateException("No product instance found for transfer");
                }

                // Create the transfer life cycle based upon the gathered information.
                var transferEvent = new TransferEvent(new TransferInformation(inLocation, outLocation));

                // Get the right data class of the product by matching against the old location, and transfer
                // productOwner
                var precedingProductChange = activeProducts.get(productInstance);
                outProductChange.setStatus(productInstance.getProductID(), precedingProductChange);
                inProductChange.setStatus(productInstance.getProductID(), outProductChange);

                // transfer owner ship
                activeProducts.put(productInstance, inProductChange);

                productInstance.addProductLocationToLifeCycle(transferEvent);
                productInstance.addPropertiesToList(outPropertyList);
                productInstance.addPropertiesToList(inPropertyList);
            }
        }
    }

    /**
     * Validate that we have exactly one IN and one OUT transfer action.
     */
    private static void validateTransferActions(List<ProductChange> inProducts, List<ProductChange> outProducts) {
        if (inProducts.size() != 1) {
            throw new IllegalStateException("Expected exactly one IN action, found %d".formatted(inProducts.size()));
        }

        if (outProducts.size() != 1) {
            throw new IllegalStateException("Expected exactly one OUT action, found %d".formatted(outProducts.size()));
        }
    }

    /**
     * End a product lifecycle.
     */
    private static void endLifeCycle(Map<ProductInstance, ProductChange> activeProducts, ProductChange prodChange) {
        // assume that the first product entering the system for this product owner also leaves the system as first
        var productInstance = getProductInstanceForSlot(activeProducts, prodChange.getProductOwner(),
                prodChange.getSlot());

        if (productInstance == null) {
            throw new IllegalStateException("No product instance found to unload");
        }

        var locationInformation = prodChange.createLocationInformation();
        var unloadEvent = new ExitEvent(locationInformation);
        prodChange.setStatus(productInstance.getProductID(), activeProducts.get(productInstance));

        productInstance.addProductLocationToLifeCycle(unloadEvent);
        activeProducts.remove(productInstance); // deactivate
    }

    /**
     * Handle a property change for a product.
     */
    private static void propertyChange(Map<ProductInstance, ProductChange> activeProducts, ProductChange prodChange) {
        // if the product change contains a slot than use it als find the most recent product change
        if (activeProducts.isEmpty()) {
            return;
        }
        var propertyList = prodChange.getChangedProperties();
        var productInstance = getProductInstanceForSlot(activeProducts, prodChange.getProductOwner(),
                prodChange.getSlot());

        if (productInstance == null) {
            if (!propertyList.isEmpty()) {
                // don't know where to assign the property changes to
                LOGGER.warn("No product instance found for property change: " + prodChange + System.lineSeparator()
                        + "Candidates:" + System.lineSeparator() + getProductLocationsString(activeProducts));
            }
            return;
        }

        prodChange.setStatus(productInstance.getProductID(), activeProducts.get(productInstance));
        activeProducts.put(productInstance, prodChange);
        productInstance.addPropertiesToList(propertyList);
    }
}
