
package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link NetworkConnection}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableNetworkConnection.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableNetworkConnection implements NetworkConnection {
  private final String containerId;
  private final @Nullable EndpointConfig endpointConfig;

  private ImmutableNetworkConnection(
      String containerId,
      @Nullable EndpointConfig endpointConfig) {
    this.containerId = containerId;
    this.endpointConfig = endpointConfig;
  }

  /**
   * @return The value of the {@code containerId} attribute
   */
  @JsonProperty("Container")
  @Override
  public String containerId() {
    return containerId;
  }

  /**
   * @return The value of the {@code endpointConfig} attribute
   */
  @JsonProperty("EndpointConfig")
  @Override
  public @Nullable EndpointConfig endpointConfig() {
    return endpointConfig;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConnection#containerId() containerId} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for containerId
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConnection withContainerId(String value) {
    String newValue = Objects.requireNonNull(value, "containerId");
    if (this.containerId.equals(newValue)) return this;
    return new ImmutableNetworkConnection(newValue, this.endpointConfig);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link NetworkConnection#endpointConfig() endpointConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for endpointConfig (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableNetworkConnection withEndpointConfig(@Nullable EndpointConfig value) {
    if (this.endpointConfig == value) return this;
    return new ImmutableNetworkConnection(this.containerId, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableNetworkConnection} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableNetworkConnection
        && equalsByValue((ImmutableNetworkConnection) another);
  }

  private boolean equalsByValue(ImmutableNetworkConnection another) {
    return containerId.equals(another.containerId)
        && Objects.equals(endpointConfig, another.endpointConfig);
  }

  /**
   * Computes a hash code from attributes: {@code containerId}, {@code endpointConfig}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + containerId.hashCode();
    h += (h << 5) + Objects.hashCode(endpointConfig);
    return h;
  }

  /**
   * Prints the immutable value {@code NetworkConnection} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "NetworkConnection{"
        + "containerId=" + containerId
        + ", endpointConfig=" + endpointConfig
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link NetworkConnection} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable NetworkConnection instance
   */
  public static ImmutableNetworkConnection copyOf(NetworkConnection instance) {
    if (instance instanceof ImmutableNetworkConnection) {
      return (ImmutableNetworkConnection) instance;
    }
    return ImmutableNetworkConnection.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableNetworkConnection ImmutableNetworkConnection}.
   * <pre>
   * ImmutableNetworkConnection.builder()
   *    .containerId(String) // required {@link NetworkConnection#containerId() containerId}
   *    .endpointConfig(org.mandas.docker.client.messages.EndpointConfig | null) // nullable {@link NetworkConnection#endpointConfig() endpointConfig}
   *    .build();
   * </pre>
   * @return A new ImmutableNetworkConnection builder
   */
  public static ImmutableNetworkConnection.Builder builder() {
    return new ImmutableNetworkConnection.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableNetworkConnection ImmutableNetworkConnection}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements NetworkConnection.Builder {
    private static final long INIT_BIT_CONTAINER_ID = 0x1L;
    private long initBits = 0x1L;

    private String containerId;
    private EndpointConfig endpointConfig;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code NetworkConnection} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(NetworkConnection instance) {
      Objects.requireNonNull(instance, "instance");
      this.containerId(instance.containerId());
      @Nullable EndpointConfig endpointConfigValue = instance.endpointConfig();
      if (endpointConfigValue != null) {
        endpointConfig(endpointConfigValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConnection#containerId() containerId} attribute.
     * @param containerId The value for containerId 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Container")
    public final Builder containerId(String containerId) {
      this.containerId = Objects.requireNonNull(containerId, "containerId");
      initBits &= ~INIT_BIT_CONTAINER_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link NetworkConnection#endpointConfig() endpointConfig} attribute.
     * @param endpointConfig The value for endpointConfig (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("EndpointConfig")
    public final Builder endpointConfig(@Nullable EndpointConfig endpointConfig) {
      this.endpointConfig = endpointConfig;
      return this;
    }

    /**
     * Builds a new {@link ImmutableNetworkConnection ImmutableNetworkConnection}.
     * @return An immutable instance of NetworkConnection
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableNetworkConnection build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableNetworkConnection(containerId, endpointConfig);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_CONTAINER_ID) != 0) attributes.add("containerId");
      return "Cannot build NetworkConnection, some of required attributes are not set " + attributes;
    }
  }
}
