//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.cif2cif;

import static org.eclipse.escet.cif.common.CifEventUtils.filterAutomata;
import static org.eclipse.escet.cif.common.CifEventUtils.filterMonitorAuts;
import static org.eclipse.escet.cif.common.CifValueUtils.createConjunction;
import static org.eclipse.escet.cif.common.CifValueUtils.createDisjunction;
import static org.eclipse.escet.cif.common.CifValueUtils.makeInverse;
import static org.eclipse.escet.common.emf.EMFHelper.deepclone;
import static org.eclipse.escet.common.java.Lists.copy;
import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Lists.listc;
import static org.eclipse.escet.common.java.Lists.set2list;
import static org.eclipse.escet.common.java.Maps.mapc;
import static org.eclipse.escet.common.java.Sets.set;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.eclipse.escet.cif.common.CifEdgeUtils;
import org.eclipse.escet.cif.common.CifEventUtils;
import org.eclipse.escet.cif.common.CifEventUtils.Alphabets;
import org.eclipse.escet.cif.common.CifLocationUtils;
import org.eclipse.escet.cif.common.CifSortUtils;
import org.eclipse.escet.cif.common.CifValueUtils;
import org.eclipse.escet.cif.metamodel.cif.automata.Automaton;
import org.eclipse.escet.cif.metamodel.cif.automata.Edge;
import org.eclipse.escet.cif.metamodel.cif.automata.EdgeEvent;
import org.eclipse.escet.cif.metamodel.cif.automata.EdgeSend;
import org.eclipse.escet.cif.metamodel.cif.automata.IfUpdate;
import org.eclipse.escet.cif.metamodel.cif.automata.Location;
import org.eclipse.escet.cif.metamodel.cif.automata.Update;
import org.eclipse.escet.cif.metamodel.cif.declarations.Event;
import org.eclipse.escet.cif.metamodel.cif.expressions.EventExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.types.VoidType;
import org.eclipse.escet.cif.metamodel.java.CifConstructors;
import org.eclipse.escet.common.java.Assert;
import org.eclipse.escet.common.java.ListProductIterator;
import org.eclipse.escet.common.java.Lists;
import org.eclipse.escet.common.java.output.WarnOutput;

/**
 * Linearization transformation that creates the Cartesian product of all edges for non-tau events, combining them in
 * all possible combinations. This results in self loops for all non-tau events, where the combination of all self loops
 * maintains all non-deterministic choices of the original specification. Worst case, the model size of the resulting
 * specification compared to the model size of the original specification could blow up exponentially. To prevent some
 * of the blow-up, use {@link LinearizeProductWithMerge}.
 *
 * <p>
 * This transformation produces linearized edges in the order that adheres to the transition execution order as defined
 * by the controller properties checker. However, since it may generate multiple linearized edges for the same event, it
 * is better to use the {@link LinearizeMerge} transformation instead, if compliance to the execution scheme of the
 * controller properties checker is desired.
 * </p>
 *
 * <p>
 * See the {@link LinearizeBase} class for further details.
 * </p>
 */
public class LinearizeProduct extends LinearizeBase {
    /**
     * Whether to try to merge edges where possible. If enabled, different alternatives of an automaton to participate
     * in synchronization are merged into one alternative if possible. Merging alternatives can help to reduce
     * exponential blow-up. Essentially, if there is no guard overlap, it then does something similar to
     * {@link LinearizeMerge}.
     */
    private final boolean tryMergeEdges;

    /**
     * Constructor for the {@link LinearizeProduct} class.
     *
     * <p>
     * Does not allow optimization of initialization of newly introduced location pointers, by analyzing declarations
     * (used for instance in initialization predicates) to see whether they have constant values.
     * </p>
     *
     * @param warnOutput Callback to send warnings to the user.
     */
    public LinearizeProduct(WarnOutput warnOutput) {
        this(false, warnOutput);
    }

    /**
     * Constructor for the {@link LinearizeProduct} class.
     *
     * @param optInits Whether to allow optimization of initialization of newly introduced location pointers, by
     *     analyzing declarations (used for instance in initialization predicates) to see whether they have constant
     *     values.
     * @param warnOutput Callback to send warnings to the user.
     */
    public LinearizeProduct(boolean optInits, WarnOutput warnOutput) {
        this(false, optInits, warnOutput);
    }

    /**
     * Constructor for the {@link LinearizeProduct} class. Is only used by {@link LinearizeProductWithMerge}.
     *
     * @param tryMergeEdges Whether to try to merge edges where possible. If enabled, different alternatives of an
     *     automaton to participate in synchronization are merged into one alternative if possible. Merging alternatives
     *     can help to reduce exponential blow-up. Essentially, if there is no guard overlap, it then does something
     *     similar to {@link LinearizeMerge}.
     * @param optInits Whether to allow optimization of initialization of newly introduced location pointers, by
     *     analyzing declarations (used for instance in initialization predicates) to see whether they have constant
     *     values.
     * @param warnOutput Callback to send warnings to the user.
     */
    LinearizeProduct(boolean tryMergeEdges, boolean optInits, WarnOutput warnOutput) {
        super(optInits, warnOutput);
        this.tryMergeEdges = tryMergeEdges;
    }

    @Override
    protected void createEdges(List<Automaton> auts, Automaton mergedAut, Location mergedLoc) {
        // Get all events to process (the merged alphabet). The events are in the order as prescribed by the transition
        // execution order of the controller properties checker.
        List<Event> events = set2list(CifEventUtils.getAlphabet(mergedAut));

        // Linearize the edges.
        List<LinearizedEdgeIterator> iterators = linearizeEdges(auts, alphabets, events, lpIntroducer, true, false,
                tryMergeEdges);

        // Add the linearized edges.
        List<Edge> edges = mergedLoc.getEdges();
        for (Iterator<Edge> iterator: iterators) {
            while (iterator.hasNext()) {
                edges.add(iterator.next());
            }
        }
    }

    /**
     * Linearizes edges. See {@link LinearizeProduct} and {@link LinearizeProductWithMerge}.
     *
     * <p>
     * Event 'tau' is treated as a special case by the {@link #mergeTauEdges} method, since monitors don't affect 'tau'
     * events, and 'tau' doesn't synchronize. That is, this method ignores 'tau' events. This method handles all other
     * events, including channels.
     * </p>
     *
     * <p>
     * This method produces linearized edges in the order that adheres to the transition execution order as defined by
     * the controller properties checker, if the {@code auts}, {@code alphabets} and {@code events} are all given in the
     * order that adheres to the transition execution order as defined by the controller properties checker.
     * </p>
     *
     * @param auts The original automata, sorted in ascending order based on their absolute names (without escaping).
     *     See also {@link CifSortUtils#sortCifObjects}. This order matches the order prescribed by the execution scheme
     *     of the controller properties checker.
     * @param alphabets Per automaton, all the alphabets.
     * @param events The events for which to create linearized edges. Usually the union of the synchronization, send,
     *     and receive alphabets of the given automata.
     * @param locPtrManager Location pointer manager.
     * @param removeMonitors Whether to remove all monitors from the given automata ({@code true}) or leave the monitors
     *     intact ({@code false}).
     * @param addLocPtrUpdates Whether to add location pointer updates.
     * @param tryMergeEdges Whether to try to merge edges where possible. If enabled, different alternatives of an
     *     automaton to participate in synchronization are merged into one alternative if possible. Merging alternatives
     *     can help to reduce exponential blow-up. Essentially, if there is no guard overlap, it then does something
     *     similar to {@link LinearizeMerge}.
     * @return Per event (in the given order), an iterator that produces linearized edges for the event.
     */
    public static List<LinearizedEdgeIterator> linearizeEdges(List<Automaton> auts, List<Alphabets> alphabets,
            List<Event> events, LocationPointerManager locPtrManager, boolean removeMonitors, boolean addLocPtrUpdates,
            boolean tryMergeEdges)
    {
        // Get alphabets.
        List<Set<Event>> syncAlphabets = listc(auts.size());
        List<Set<Event>> sendAlphabets = listc(auts.size());
        List<Set<Event>> recvAlphabets = listc(auts.size());
        List<Set<Event>> moniAlphabets = listc(auts.size());
        for (Alphabets autAlphabets: alphabets) {
            syncAlphabets.add(autAlphabets.syncAlphabet);
            sendAlphabets.add(autAlphabets.sendAlphabet);
            recvAlphabets.add(autAlphabets.recvAlphabet);
            moniAlphabets.add(autAlphabets.moniAlphabet);
        }

        // Filter automata, per event.
        List<List<Automaton>> syncAuts;
        List<List<Automaton>> sendAuts;
        List<List<Automaton>> recvAuts;
        List<Set<Automaton>> moniAuts;
        syncAuts = filterAutomata(auts, syncAlphabets, events);
        sendAuts = filterAutomata(auts, sendAlphabets, events);
        recvAuts = filterAutomata(auts, recvAlphabets, events);
        moniAuts = filterMonitorAuts(auts, moniAlphabets, events);

        // Remove monitors from all automata.
        if (removeMonitors) {
            for (Automaton aut: auts) {
                aut.setMonitors(null);
            }
        }

        // Create per event an iterator for the linearized edges, and return all iterators.
        List<LinearizedEdgeIterator> iterators = listc(events.size());
        for (int i = 0; i < events.size(); i++) {
            LinearizedEdgeIterator linearizedEdgesIter = linearizeEdges(events.get(i), syncAuts.get(i), sendAuts.get(i),
                    recvAuts.get(i), moniAuts.get(i), locPtrManager, addLocPtrUpdates, tryMergeEdges);
            iterators.add(linearizedEdgesIter);
        }
        return iterators;
    }

    /**
     * Linearizes edges for the given event to all possible combinations.
     *
     * @param event The event.
     * @param syncAuts The automata that synchronize over the event, sorted in ascending order based on their absolute
     *     names (without escaping). See also {@link CifSortUtils#sortCifObjects}.
     * @param sendAuts The automata that send over the event, sorted in ascending order based on their absolute names
     *     (without escaping). See also {@link CifSortUtils#sortCifObjects}. This order matches the order prescribed by
     *     the execution scheme of the controller properties checker.
     * @param recvAuts The automata that receive over the event, sorted in ascending order based on their absolute names
     *     (without escaping). See also {@link CifSortUtils#sortCifObjects}. This order matches the order prescribed by
     *     the execution scheme of the controller properties checker.
     * @param moniAuts The automata that monitor the event, sorted in ascending order based on their absolute names
     *     (without escaping). See also {@link CifSortUtils#sortCifObjects}.
     * @param locPtrManager Location pointer manager.
     * @param addLocPtrUpdates Whether to add location pointer updates.
     * @param tryMergeEdges Whether to try to merge edges where possible. If enabled, different alternatives of an
     *     automaton to participate in synchronization are merged into one alternative if possible. Merging alternatives
     *     can help to reduce exponential blow-up. Essentially, if there is no guard overlap, it then does something
     *     similar to {@link LinearizeMerge}.
     * @return An iterator that produces linearized edges for the given event.
     */
    private static LinearizedEdgeIterator linearizeEdges(Event event, List<Automaton> syncAuts,
            List<Automaton> sendAuts, List<Automaton> recvAuts, Set<Automaton> moniAuts,
            LocationPointerManager locPtrManager, boolean addLocPtrUpdates, boolean tryMergeEdges)
    {
        // Is the event a channel?
        boolean isChannel = event.getType() != null;
        boolean isVoid = isChannel && (event.getType() instanceof VoidType);

        // Get the alternative synchronization options, per synchronizing automaton. We consider the automata in the
        // given order, and the locations and edges in model order, to ensure that we adhere to the transition execution
        // order as defined by the controller properties checker.
        List<List<Participation>> autsSyncOptions = listc(syncAuts.size());
        for (Automaton aut: syncAuts) {
            // Check whether automaton is a monitor automaton for the event.
            boolean monitor = moniAuts.contains(aut);

            // Get edge-event and monitored edge-event synchronization options of the automaton, for the event. Also get
            // the monitored locations of the automaton (where a monitor edge is needed).
            List<Participation> autSyncOptions = list();
            List<Location> monitoredLocs = monitor ? listc(aut.getLocations().size()) : List.of();
            List<EdgeEvent> locEdgeEvents = list();
            for (Location loc: aut.getLocations()) {
                // Collect edge events on edges of this location, for the event. Also detect whether there is one with
                // a trivially 'true' guard.
                boolean trueGuard = false;
                locEdgeEvents.clear();
                for (Edge edge: loc.getEdges()) {
                    for (EdgeEvent edgeEvent: edge.getEvents()) {
                        Event evt = CifEventUtils.getEventFromEdgeEvent(edgeEvent);
                        if (evt == event) {
                            // If 'true' guard, monitor has 'false' guard.
                            if (edge.getGuards().isEmpty()) {
                                trueGuard = true;
                            }

                            // Collect the edge event.
                            locEdgeEvents.add(edgeEvent);
                        }
                    }
                }

                // Mark location as needing monitor edge, if event is monitored, and there is no outgoing edge for the
                // event with a trivially 'true' guard.
                boolean locNeedsMonitorEdge = monitor && !trueGuard;

                // If the location needs an implicit monitor edge and has exactly one explicit edge event for the event,
                // then merge the monitor edge into the explicit edge, to reduce the number of combinations in the
                // product, preventing blow-ups. If an edge has multiple edge events for the event, this essentially
                // counts as multiple edges for the event.
                if (tryMergeEdges && locNeedsMonitorEdge && locEdgeEvents.size() == 1) {
                    // Add the monitored edge event as a synchronization option. This handles both the edge event
                    // and it needing a monitor, thus handling all synchronization options for this location.
                    EdgeEvent locEdgeEvent = locEdgeEvents.get(0);
                    autSyncOptions.add(new MonitoredEdgeEventSyncParticipation(locEdgeEvent));
                    continue;
                }

                // Add the explicit edge events of the location as synchronization options.
                for (EdgeEvent locEdgeEvent: locEdgeEvents) {
                    autSyncOptions.add(new EdgeEventParticipation(locEdgeEvent));
                }

                // If applicable, add the location to the list of locations that require a monitor edge.
                if (locNeedsMonitorEdge) {
                    monitoredLocs.add(loc);
                }
            }

            // Add monitored locations synchronization option, if needed for at least one location.
            if (!monitoredLocs.isEmpty()) {
                autSyncOptions.add(
                        new MonitoredLocsSyncParticipation(Collections.unmodifiableList(monitoredLocs), event));
            }

            // Combine synchronization options with different source locations, if they are the only options in their
            // source locations. In those cases we can combine them without having overlapping guards. Combining the
            // options reduces the number of combinations in the product, preventing blow-ups.
            if (tryMergeEdges) {
                combineSingleOptionPerSrcLocAutSyncOptions(autSyncOptions, aut, event);
            }

            // Store synchronization options for this automaton.
            autsSyncOptions.add(autSyncOptions);
        }

        // Get the participation options for send edge events. We consider the automata in the given order, and the
        // locations, edges and edge events in model order, to ensure that we adhere to the transition execution order
        // as defined by the controller properties checker.
        List<Participation> sendOptions = list();
        for (Automaton aut: sendAuts) {
            for (Location loc: aut.getLocations()) {
                for (Edge edge: loc.getEdges()) {
                    for (EdgeEvent edgeEvent: edge.getEvents()) {
                        Event evt = CifEventUtils.getEventFromEdgeEvent(edgeEvent);
                        if (evt == event) {
                            sendOptions.add(new EdgeEventParticipation(edgeEvent));
                        }
                    }
                }
            }
        }

        // Get the participation options for receive edge events. We consider the automata in the given order, and the
        // locations, edges and edge events in model order, to ensure that we adhere to the transition execution order
        // as defined by the controller properties checker.
        List<Participation> recvOptions = list();
        for (Automaton aut: recvAuts) {
            for (Location loc: aut.getLocations()) {
                for (Edge edge: loc.getEdges()) {
                    for (EdgeEvent edgeEvent: edge.getEvents()) {
                        Event evt = CifEventUtils.getEventFromEdgeEvent(edgeEvent);
                        if (evt == event) {
                            recvOptions.add(new EdgeEventParticipation(edgeEvent));
                        }
                    }
                }
            }
        }

        // Create and return an iterator for all combinations of participations.
        List<List<Participation>> possibilities = autsSyncOptions;
        if (isChannel) {
            possibilities.add(sendOptions);
            possibilities.add(recvOptions);
        }

        ListProductIterator<Participation> combinationsIter = new ListProductIterator<>(possibilities);
        return new LinearizedEdgeIterator(event, isChannel, isVoid, combinationsIter, locPtrManager, addLocPtrUpdates);
    }

    /**
     * Combine synchronization options for a certain event, with different source locations of the same automaton, if
     * they are the only options in their source locations.
     *
     * @param autSyncOptions The synchronization options of the automaton, for the event. Is modified in-place.
     * @param aut The automaton.
     * @param event The event.
     */
    private static void combineSingleOptionPerSrcLocAutSyncOptions(List<Participation> autSyncOptions, Automaton aut,
            Event event)
    {
        // Find the participation options per source location.
        Map<Location, List<Participation>> autSyncOptionsPerSrcLoc = mapc(aut.getLocations().size());
        for (Participation autSyncOption: autSyncOptions) {
            for (Location srcLoc: autSyncOption.getSourceLocations()) {
                autSyncOptionsPerSrcLoc.computeIfAbsent(srcLoc, k -> listc(1)).add(autSyncOption);
            }
        }

        // Find the participation options to combine, so those with exactly one option per source location. Ensure
        // that participations that concern multiple such source locations are included only once.
        List<Participation> autSyncOptionsToCombine = Lists.set2list(autSyncOptionsPerSrcLoc.entrySet().stream()
                .map(entry -> entry.getValue()).filter(ps -> ps.size() == 1).map(ps -> Lists.single(ps))
                .collect(Collectors.toCollection(() -> set())));

        // If there are multiple options to combine, combine them and remove the original non-combined options.
        if (autSyncOptionsToCombine.size() > 1) {
            // If a 'monitored source locations' synchronization option is merged, split it to partition the combined
            // and non-combined source locations. Note that there is only at most one such option per automaton.
            MonitoredLocsSyncParticipation monitoredLocsOption = autSyncOptionsToCombine.stream()
                    .filter(MonitoredLocsSyncParticipation.class::isInstance)
                    .map(MonitoredLocsSyncParticipation.class::cast).findAny().orElseGet(() -> null);
            if (monitoredLocsOption != null) {
                // Partition the source locations of the 'monitored source locations' synchronization option.
                List<Location> combinedSrcLocs = autSyncOptionsPerSrcLoc.entrySet().stream()
                        .filter(entry -> entry.getValue().contains(monitoredLocsOption))
                        .map(entry -> entry.getKey()).toList();

                List<Location> remainingSrcLocs = copy(monitoredLocsOption.locs);
                remainingSrcLocs.removeAll(combinedSrcLocs);

                // Update the synchronization options.
                autSyncOptions.remove(monitoredLocsOption);
                if (!remainingSrcLocs.isEmpty()) {
                    autSyncOptions.add(new MonitoredLocsSyncParticipation(
                            Collections.unmodifiableList(remainingSrcLocs), event));
                }

                autSyncOptionsToCombine.replaceAll(p -> (p == monitoredLocsOption)
                        ? new MonitoredLocsSyncParticipation(Collections.unmodifiableList(combinedSrcLocs), event) : p);
            }

            // Remove the original non-combined options and add the new combined one. The order between the kept options
            // and the new combined one is not relevant to the execution scheme: you are always in one location at a
            // time, and each of the combined locations has only one synchronization option, so kept choices and the
            // combined one are never enabled at the same time.
            autSyncOptions.removeAll(autSyncOptionsToCombine);
            autSyncOptions.add(new CombinedSyncParticipation(autSyncOptionsToCombine));
        }
    }

    /** Iterator that lazily creates the linearized edges one-by-one as they are {@link Iterator#next requested}. */
    public static class LinearizedEdgeIterator implements Iterator<Edge> {
        /** The event. */
        private final Event event;

        /** Whether the event is a channel. */
        private final boolean isChannel;

        /** Whether the event is a void channel. */
        private final boolean isVoid;

        /**
         * An iterator over all possible combinations of participations. Each combination represents a linearized edge.
         */
        private final ListProductIterator<Participation> combinationsIter;

        /** The location pointer manager to use. */
        private final LocationPointerManager locPtrManager;

        /** Whether to add location pointer updates. */
        private final boolean addLocPtrUpdates;

        /**
         * Constructor for the {@link LinearizedEdgeIterator} class.
         *
         * @param event The event.
         * @param isChannel Whether the event is a channel.
         * @param isVoid Whether the event is a void channel.
         * @param combinationsIter An iterator over all possible combinations of participations. Each combination
         *     represents a linearized edge.
         * @param locPtrManager The location pointer manager to use.
         * @param addLocPtrUpdates Whether to add location pointer updates.
         */
        private LinearizedEdgeIterator(Event event, boolean isChannel, boolean isVoid,
                ListProductIterator<Participation> combinationsIter, LocationPointerManager locPtrManager,
                boolean addLocPtrUpdates)
        {
            this.event = event;
            this.isChannel = isChannel;
            this.isVoid = isVoid;
            this.combinationsIter = combinationsIter;
            this.locPtrManager = locPtrManager;
            this.addLocPtrUpdates = addLocPtrUpdates;
        }

        /**
         * Returns the result size of the iterator, i.e., the number of linearized edges that the iterator will iterate
         * over and return by the {@link #next} method. Note that the result size does <em>not</em> represent the number
         * of <em>remaining</em> edges, but rather all <em>possible</em> ones, and therefore the result size remains
         * constant even during and after iterating over this iterator.
         *
         * @return The result size of the iterator, if it can be represented as a long value.
         */
        public Optional<Long> getResultSize() {
            return combinationsIter.getResultSize();
        }

        @Override
        public boolean hasNext() {
            return combinationsIter.hasNext();
        }

        @Override
        public Edge next() {
            // Get next combination, with for each automaton the way it participates.
            List<Participation> participations = combinationsIter.next();

            // Create combined linearized guard.
            List<Expression> guards = listc(2 * participations.size()); // Optimized for location pointer and guard.
            for (Participation participation: participations) {
                addGuards(participation, guards, locPtrManager);
            }
            Expression guard = createConjunction(guards);

            // Get combined updates.
            List<Update> updates = list();
            List<Update> rcvUpdates = list();
            for (int i = 0; i < participations.size(); i++) {
                Participation participation = participations.get(i);

                if (isChannel && !isVoid && i == participations.size() - 1) {
                    addUpdates(participation, rcvUpdates, locPtrManager, addLocPtrUpdates);
                } else {
                    addUpdates(participation, updates, locPtrManager, addLocPtrUpdates);
                }
            }

            // Handle non-void communication.
            if (isChannel && !isVoid) {
                // Get send value.
                int sendIdx = participations.size() - 2;
                Participation sendParticipation = participations.get(sendIdx);
                EdgeEvent sendEdgeEvent = ((EdgeEventParticipation)sendParticipation).edgeEvent;
                Expression sendValue = ((EdgeSend)sendEdgeEvent).getValue();

                // Replace received values by the send value.
                rcvUpdates = replaceUpdates(rcvUpdates, sendValue);
                updates.addAll(rcvUpdates);
            }

            // Ignore target locations, as location pointer updates already handle that.

            // Create new linearized edge.
            EventExpression eventRef = CifConstructors.newEventExpression();
            eventRef.setEvent(event);
            eventRef.setType(CifConstructors.newBoolType());

            EdgeEvent edgeEvent = CifConstructors.newEdgeEvent();
            edgeEvent.setEvent(eventRef);

            Edge edge = CifConstructors.newEdge();
            edge.getEvents().add(edgeEvent);
            edge.getGuards().add(guard);
            edge.getUpdates().addAll(updates);

            // Return the linearized edge.
            return edge;
        }

        /**
         * Adds guards for the given participation.
         *
         * @param participation The participation for which to add guards.
         * @param guards The guards so far. Is extended in-place.
         * @param locPtrManager Location pointer manager.
         */
        private static void addGuards(Participation participation, List<Expression> guards,
                LocationPointerManager locPtrManager)
        {
            if (participation instanceof EdgeEventParticipation eeParticipation) {
                // Get source location.
                EdgeEvent edgeEvent = eeParticipation.edgeEvent;
                Edge edge = (Edge)edgeEvent.eContainer();
                Location src = CifEdgeUtils.getSource(edge);

                // Add source location reference as guard.
                Automaton aut = (Automaton)src.eContainer();
                if (aut.getLocations().size() > 1) {
                    Expression srcRef = locPtrManager.createLocRef(src);
                    guards.add(srcRef);
                }

                // Add edge guards.
                guards.addAll(deepclone(edge.getGuards()));
            } else if (participation instanceof MonitoredEdgeEventSyncParticipation meesParticipation) {
                // Get source location.
                EdgeEvent edgeEvent = meesParticipation.edgeEvent;
                Edge edge = (Edge)edgeEvent.eContainer();
                Location src = CifEdgeUtils.getSource(edge);

                // Add source location reference as guard.
                Automaton aut = (Automaton)src.eContainer();
                if (aut.getLocations().size() > 1) {
                    Expression srcRef = locPtrManager.createLocRef(src);
                    guards.add(srcRef);
                }
            } else if (participation instanceof MonitoredLocsSyncParticipation mlsParticipation) {
                // Get source locations and event.
                List<Location> srcLocs = mlsParticipation.locs;
                Event event = mlsParticipation.monitoredEvent;

                // Add guards, per location.
                List<Expression> locsGuards = listc(srcLocs.size());
                for (Location srcLoc: srcLocs) {
                    // Initialization.
                    List<Expression> locGuards = listc(4);

                    // Add source location reference as guard.
                    Automaton aut = CifLocationUtils.getAutomaton(srcLoc);
                    if (aut.getLocations().size() > 1) {
                        Expression srcRef = locPtrManager.createLocRef(srcLoc);
                        locGuards.add(srcRef);
                    }

                    // Add monitor edge guards.
                    for (Edge srcEdge: srcLoc.getEdges()) {
                        for (EdgeEvent srcEdgeEvent: srcEdge.getEvents()) {
                            Event evt = CifEventUtils.getEventFromEdgeEvent(srcEdgeEvent);
                            if (evt == event) {
                                List<Expression> edgeGuards = deepclone(srcEdge.getGuards());
                                Expression edgeGuard = createConjunction(edgeGuards);
                                locGuards.add(makeInverse(edgeGuard));
                            }
                        }
                    }

                    // Combine guards for the location.
                    locsGuards.add(createConjunction(locGuards));
                }

                // Combine guards for the automaton.
                guards.add(createDisjunction(locsGuards));
            } else if (participation instanceof CombinedSyncParticipation csParticipation) {
                // Let each participant add its guards and merge those per participant to a single guard using
                // conjunction.
                List<Expression> subsGuards = listc(csParticipation.subParticipations.size());
                for (Participation subParticipation: csParticipation.subParticipations) {
                    List<Expression> subGuards = listc(1); // Optimized for a single guard.
                    addGuards(subParticipation, subGuards, locPtrManager);
                    subsGuards.add(CifValueUtils.createConjunction(subGuards));
                }

                // Merge the participant guards using disjunction to form the combined guard, and add it as the final
                // guard.
                guards.add(CifValueUtils.createDisjunction(subsGuards));
            } else {
                throw new AssertionError("Unknown participation: " + participation);
            }
        }

        /**
         * Adds updates for the given participation.
         *
         * @param participation The participation for which to add updates.
         * @param updates The updates so far. Is extended in-place.
         * @param locPtrManager The location pointer manager to use.
         * @param addLocPtrUpdate Whether to add a location pointer update.
         */
        private static void addUpdates(Participation participation, List<Update> updates,
                LocationPointerManager locPtrManager, boolean addLocPtrUpdate)
        {
            if (participation instanceof EdgeEventParticipation eeParticipation) {
                // Add original edge updates.
                EdgeEvent edgeEvent = eeParticipation.edgeEvent;
                Edge edge = (Edge)edgeEvent.eContainer();
                updates.addAll(deepclone(edge.getUpdates()));

                // Add location pointer updates.
                if (addLocPtrUpdate) {
                    Location srcLoc = CifEdgeUtils.getSource(edge);
                    Location tgtLoc = CifEdgeUtils.getTarget(edge);
                    if (srcLoc != tgtLoc) {
                        updates.add(locPtrManager.createLocUpdate(tgtLoc));
                    }
                }
            } else if (participation instanceof MonitoredEdgeEventSyncParticipation meesParticipation) {
                // Get the edge.
                EdgeEvent edgeEvent = meesParticipation.edgeEvent;
                Edge edge = (Edge)edgeEvent.eContainer();

                // See if we need to add a location pointer update, if requested.
                Location srcLoc = CifEdgeUtils.getSource(edge);
                Location tgtLoc = CifEdgeUtils.getTarget(edge);
                boolean needsLocPtrUpdate = addLocPtrUpdate && srcLoc != tgtLoc;

                // If needed, add the required updates, inside an 'if' update with the required guards.
                if (needsLocPtrUpdate || !edge.getUpdates().isEmpty()) {
                    // Create a new 'if' update, and add it as the update for this participation.
                    IfUpdate ifUpd = CifConstructors.newIfUpdate();
                    updates.add(ifUpd);

                    // Add the edge guards to the 'if' update. We don't need a source location guard for the 'if'
                    // update, since this participant concerns only a single source location, and guard of the
                    // linearized edge checks for being in that source location already, and thus we can assume we're
                    // still there when doing the updates.
                    ifUpd.getGuards().addAll(deepclone(edge.getGuards()));

                    // Copy the original updates to the 'if' update.
                    ifUpd.getThens().addAll(deepclone(edge.getUpdates()));

                    // Add a location pointer update to the 'if' update, if needed.
                    if (needsLocPtrUpdate) {
                        ifUpd.getThens().add(locPtrManager.createLocUpdate(tgtLoc));
                    }
                }
            } else if (participation instanceof MonitoredLocsSyncParticipation) {
                // No updates.
            } else if (participation instanceof CombinedSyncParticipation csParticipation) {
                // Create an 'if' update, with for each sub-participant its updates, as follows:
                // 'if <src-locs-0>: <updates-0> elif <src-locs-1>: <updates-1> ... end'.
                IfUpdate ifUpd = CifConstructors.newIfUpdate();
                for (Participation subParticipation: csParticipation.subParticipations) {
                    // Get the updates for the sub-participant.
                    List<Update> subUpdates = list();
                    addUpdates(subParticipation, subUpdates, locPtrManager, addLocPtrUpdate);

                    // If there are no updates for this participant, skip it.
                    if (subUpdates.isEmpty()) {
                        continue;
                    }

                    // Construct the if/elif guard.
                    List<Expression> srcLocRefs = subParticipation.getSourceLocations().stream()
                            .map(locPtrManager::createLocRef).toList();
                    Expression guard = CifValueUtils.createDisjunction(srcLocRefs);

                    // If the sub-participant's updates consist of exactly one 'if' update without 'elif's and 'else',
                    // merge it into the one we're constructing here.
                    if (subUpdates.size() == 1 && subUpdates.get(0) instanceof IfUpdate subIfUpd
                            && subIfUpd.getElifs().isEmpty() && subIfUpd.getElses().isEmpty())
                    {
                        guard = CifValueUtils.createConjunction(
                                List.of(guard, CifValueUtils.createConjunction(subIfUpd.getGuards())));
                        subUpdates = subIfUpd.getThens();
                    }

                    // Add the guard and updates to the 'if' update, as 'if' guard/thens, or as 'elif' with guard/thens.
                    if (ifUpd.getGuards().isEmpty()) {
                        ifUpd.getGuards().add(guard);
                        ifUpd.getThens().addAll(subUpdates);
                    } else {
                        ifUpd.getElifs().add(CifConstructors.newElifUpdate(List.of(guard), null, subUpdates));
                    }
                }

                // If at least one of the participants has updates, add the new 'if' update as the final update.
                if (!ifUpd.getGuards().isEmpty()) {
                    updates.add(ifUpd);
                }
            } else {
                throw new AssertionError("Unknown participation: " + participation);
            }
        }
    }

    /** A participation in the product. */
    private static interface Participation {
        /**
         * Returns the source locations of the participation.
         *
         * @return The source locations.
         */
        List<Location> getSourceLocations();
    }

    /**
     * A synchronization, send or receive participation for an edge event.
     *
     * @param edgeEvent The edge event that participates. Can be a synchronization, send or receive edge event.
     */
    private static record EdgeEventParticipation(EdgeEvent edgeEvent) implements Participation {
        @Override
        public List<Location> getSourceLocations() {
            Edge edge = (Edge)edgeEvent.eContainer();
            Location src = CifEdgeUtils.getSource(edge);
            return List.of(src);
        }
    }

    /**
     * A synchronization participation for an edge event that is monitored. The option represents the merge of the edge
     * of the edge event and its implicit monitor edge. The edge event is the only edge event for its corresponding
     * event on any edge of its source location.
     *
     * @param edgeEvent The edge event.
     */
    private static record MonitoredEdgeEventSyncParticipation(EdgeEvent edgeEvent) implements Participation {
        @Override
        public List<Location> getSourceLocations() {
            Edge edge = (Edge)edgeEvent.eContainer();
            Location src = CifEdgeUtils.getSource(edge);
            return List.of(src);
        }
    }

    /**
     * A synchronization participation for one or more monitored locations of a single automaton. The monitored
     * locations have no edge for the monitored event, or it cannot be statically determined that the existing edges
     * cover all cases (always enable the event in the location).
     *
     * <p>
     * <strong>Example</strong>
     * </p>
     * <p>
     * Consider the following locations 'loc1' and 'loc2', for event 'e':
     * </p>
     * <pre>
     * location loc1:
     *     edge e when g1 ...
     *     edge e when g2 ...
     * location loc2:
     *     edge e when g3
     *     edge e when g4
     * </pre>
     * <p>
     * Then this synchronization represents the following monitor edge:
     * </p>
     * <pre>
     * edge e when (loc1 and not g1 and not g2) or (loc2 and not g3 and not g4)
     * </pre>
     *
     * @param locs The locations.
     * @param monitoredEvent The monitored event.
     */
    private static record MonitoredLocsSyncParticipation(List<Location> locs, Event monitoredEvent)
            implements Participation
    {
        /** Constructor for the {@link MonitoredLocsSyncParticipation} record. */
        public MonitoredLocsSyncParticipation {
            Assert.check(!locs.isEmpty());
        }

        @Override
        public List<Location> getSourceLocations() {
            return locs;
        }
    }

    /**
     * A synchronization participation that combines multiple other synchronization participations.
     *
     * <p>
     * The following restrictions apply:
     * </p>
     * <ul>
     * <li>Each sub-participation is not a {@link CombinedSyncParticipation}.</li>
     * <li>Each sub-participation is from a different source location of the same automaton (or represents multiple of
     * those).</li>
     * <li>Each sub-participation is the only synchronization participation for its source location (or represents
     * multiple of those).</li>
     * </ul>
     *
     * @param subParticipations The sub-participations. There are at least two of them.
     */
    private static record CombinedSyncParticipation(List<Participation> subParticipations) implements Participation {
        /** Constructor for the {@link CombinedSyncParticipation} record. */
        public CombinedSyncParticipation {
            // At least two sub-participations get combined.
            Assert.check(subParticipations.size() > 1);

            // No nesting.
            Assert.check(subParticipations.stream().allMatch(p -> !(p instanceof CombinedSyncParticipation)));

            // No overlap in source locations of the participants.
            Assert.check(subParticipations.stream().flatMap(p -> p.getSourceLocations().stream()).map(set()::add)
                    .allMatch(b -> b));
        }

        @Override
        public List<Location> getSourceLocations() {
            // Note that there is no overlap between the source locations of the different participations, so all
            // elements of the returned list are unique.
            return subParticipations.stream().flatMap(sub -> sub.getSourceLocations().stream()).toList();
        }
    }
}
