//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2024, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.conversion.preconditions;

import org.eclipse.escet.cif.checkers.CifCheckNoCompDefInst;
import org.eclipse.escet.cif.checkers.CifCheckViolations;
import org.eclipse.escet.cif.common.CifInternalFuncUtils;
import org.eclipse.escet.cif.common.CifTypeUtils;
import org.eclipse.escet.cif.metamodel.cif.declarations.DiscVariable;
import org.eclipse.escet.cif.metamodel.cif.declarations.InputVariable;
import org.eclipse.escet.cif.metamodel.cif.functions.InternalFunction;
import org.eclipse.escet.cif.metamodel.cif.types.BoolType;
import org.eclipse.escet.cif.metamodel.cif.types.CifType;
import org.eclipse.escet.cif.metamodel.cif.types.EnumType;
import org.eclipse.escet.cif.metamodel.cif.types.IntType;

/**
 * CIF check that allows discrete variables, input variables, function parameters, function local variables and function
 * return types only if they have/are a supported type:
 * <ul>
 * <li>Boolean type.</li>
 * <li>Ranged integer type with at most {@link Integer#MAX_VALUE} possible values.</li>
 * <li>Enumeration type.</li>
 * </ul>
 */
public class CifToBddVarOnlySpecificTypesCheck extends CifCheckNoCompDefInst {
    @Override
    protected void preprocessDiscVariable(DiscVariable discVar, CifCheckViolations violations) {
        if (CifInternalFuncUtils.isFuncParam(discVar)) {
            checkType(discVar.getType(), "Function parameter has", violations);
        } else if (CifInternalFuncUtils.isFuncLocalVar(discVar)) {
            checkType(discVar.getType(), "Function local variable has", violations);
        } else {
            checkType(discVar.getType(), "Discrete variable has", violations);
        }
    }

    @Override
    protected void preprocessInputVariable(InputVariable inputVar, CifCheckViolations violations) {
        checkType(inputVar.getType(), "Input variable has", violations);
    }

    @Override
    protected void preprocessInternalFunction(InternalFunction func, CifCheckViolations violations) {
        for (CifType returnType: func.getReturnTypes()) {
            checkType(returnType, "Function return type is", violations);
        }
    }

    /**
     * Check the given type.
     *
     * @param type The type to check.
     * @param prefix The prefix to use for violation messages.
     * @param violations The violations reported so far.
     */
    private void checkType(CifType type, String prefix, CifCheckViolations violations) {
        CifType normType = CifTypeUtils.normalizeType(type);

        if (normType instanceof BoolType) {
            // OK.
        } else if (normType instanceof IntType intType) {
            if (CifTypeUtils.isRangeless(intType)) {
                violations.add(type, prefix + " a rangeless integer type");
            } else {
                long count = CifTypeUtils.getPossibleValuesCount(intType);
                if (count > Integer.MAX_VALUE) {
                    violations.add(type, prefix + " an integer type with more than 2,147,483,647 possible values");
                }
            }
        } else if (normType instanceof EnumType) {
            // OK.
        } else {
            violations.add(type, prefix + " a type that is not a boolean, integer or enumeration type");
        }
    }
}
